<?php

namespace App;

use App\Contracts\SeoKeywordMapper;
use Mtc\MercuryDataModels\SeoData;
use Mtc\MercuryDataModels\SeoDefault;

trait HasSeoData
{
    /**
     * Get Seo Data for request based off section/matching
     *
     * @param string $uri
     * @param string $section
     * @param SeoKeywordMapper $mapper
     * @return array
     */
    public function getSeoData(string $uri, string $section, SeoKeywordMapper $mapper): array
    {
        if (($uriSpecificSeo = $this->getPathSpecificSeo($uri)) !== null) {
            return $this->checkForNullValuesOnSeo($uriSpecificSeo, $section, $mapper);
        }

        return $this->getSeoFromSectionDefaults($section, $mapper);
    }

    /**
     * Get SEO data that is specific to path
     *
     * @param string $uriPath
     * @return array|null
     */
    protected function getPathSpecificSeo(string $uriPath): ?array
    {
        return SeoData::query()
            ->where('path', $uriPath)
            ->first()
            ?->only([
                'title',
                'description',
                'heading'
            ]);
    }

    /**
     * Get data from defaults
     *
     * @param string $section
     * @param SeoKeywordMapper $mapper
     * @return array
     */
    protected function getSeoFromSectionDefaults(string $section, SeoKeywordMapper $mapper): array
    {
        if (($defaults = SeoDefault::query()->where('section', $section)->first()) === null) {
            return [
                'title' => '',
                'description' => '',
            ];
        }

        return [
            'title' => $mapper->map($defaults->title ?? ''),
            'description' => $mapper->map($defaults->description ?? ''),
        ];
    }

    /**
     * Check if Seo data is partially or fully filled with empty (null) values
     *
     * @param array $data
     * @param string $section
     * @param SeoKeywordMapper $mapper
     * @return array
     */
    protected function checkForNullValuesOnSeo(array $data, string $section, SeoKeywordMapper $mapper): array
    {
        if (empty($data['title']) || empty($data['description'])) {
            $defaults = $this->getSeoFromSectionDefaults($section, $mapper);

            $data['title'] = $data['title'] ?: $defaults['title'];
            $data['description'] = $data['description'] ?: $defaults['description'];
        }

        return $data;
    }
}
