<?php

namespace App\Providers;

use App\Facades\Settings;
use App\Newsletter\CampaignMonitor;
use App\Newsletter\MailChimp;
use App\SettingRepository;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\ServiceProvider;
use Illuminate\Database\Eloquent\Relations\Relation;
use Mtc\MercuryDataModels\VehicleMake;
use Mtc\MercuryDataModels\VehicleModel;
use Mtc\MercuryDataModels\VehicleOffer;

class AppServiceProvider extends ServiceProvider
{
    /**
     * Register any application services.
     *
     * @return void
     */
    public function register()
    {
        if ($this->app->runningUnitTests()) {
            $this->setMigrationLocation();
        }
    }

    /**
     * Bootstrap any application services.
     *
     * @return void
     */
    public function boot()
    {
        $this->bindNewsletterDrivers();
        $this->app->bind('site-settings', SettingRepository::class);

        Relation::morphMap([
            'make' => VehicleMake::class,
            'model' => VehicleModel::class,
            'offer' => VehicleOffer::class,
        ]);
    }

    /**
     * Set up bindings for newsletter drivers
     *
     * @return void
     */
    private function bindNewsletterDrivers()
    {
        $this->app->bind(MailChimp::class, function () {
            return new MailChimp(new \DrewM\MailChimp\MailChimp(Settings::get('newsletters.mailchimp.api_key')));
        });
        $this->app->bind(CampaignMonitor::class, function () {
            return new CampaignMonitor(new \CS_REST_Subscribers(
                Settings::get('newsletters.campaign-monitor.list_id'),
                ['api_key' => Settings::get('newsletters.campaign-monitor.api_key')]
            ));
        });
    }

    /**
     * Due to the shared data models migrations are moved in mtcmedia/mercury-data-models
     *
     * @return void
     */
    private function setMigrationLocation()
    {
        $this->app->useDatabasePath($this->app->basePath('vendor/mtcmedia/mercury-data-models/database'));
        Config::set(
            'tenancy.migration_parameters.--path',
            $this->app->basePath('vendor/mtcmedia/mercury-data-models/database/migrations/tenant')
        );
    }
}
