<?php

namespace Tests\Feature;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Mtc\Crm\Models\Enquiry;
use Mtc\Crm\Models\Form;
use Mtc\Crm\Models\FormQuestion;
use Tests\TestCase;

class EnquiryControllerTest extends TestCase
{
    use RefreshDatabase;

    protected $tenancy = true;

    public function testSubmitEnquiry()
    {
        $form = Form::factory()->create();
        $first_name = FormQuestion::factory()->create([
            'form_id' => $form->id,
            'type' => 'text',
            'name' => 'First Name',
            'validation' => 'required',
        ]);
        $last_name = FormQuestion::factory()->create([
            'form_id' => $form->id,
            'type' => 'text',
            'name' => 'Last Name',
            'validation' => 'required',
        ]);
        $email = FormQuestion::factory()->create([
            'form_id' => $form->id,
            'type' => 'email',
            'name' => 'Email',
            'validation' => 'required|email',
        ]);
        $phone = FormQuestion::factory()->create([
            'form_id' => $form->id,
            'type' => 'text',
            'name' => 'Phone Number',
            'validation' => 'required',
        ]);

        $message = FormQuestion::factory()->create([
            'form_id' => $form->id,
            'type' => 'textarea',
            'name' => 'Message',
            'validation' => 'required',
        ]);

        $response = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('enquiries.store'), [
                'form_id' => $form->id,
                'questions' => [
                    $first_name->id => 'Peter',
                    $last_name->id => 'Parker',
                    $email->id => 'john@example.com',
                    $phone->id => '12331231232',
                    $message->id => 'Loprem ipsum dolor sit amet',

                ]
            ]);
        $response->assertStatus(200);

        $this->assertTrue(Enquiry::query()->where('form_id', $form->id)->exists());
    }

    public function testMissingFields()
    {
        $form = Form::factory()->create();
        $first_name = FormQuestion::factory()->create([
            'form_id' => $form->id,
            'type' => 'text',
            'name' => 'First Name',
            'validation' => 'required',
        ]);
        $last_name = FormQuestion::factory()->create([
            'form_id' => $form->id,
            'type' => 'text',
            'name' => 'Last Name',
            'validation' => 'required',
        ]);
        $email = FormQuestion::factory()->create([
            'form_id' => $form->id,
            'type' => 'email',
            'name' => 'Email',
            'validation' => 'required|email',
        ]);
        $phone = FormQuestion::factory()->create([
            'form_id' => $form->id,
            'type' => 'text',
            'name' => 'Phone Number',
            'validation' => 'required',
        ]);

        $message = FormQuestion::factory()->create([
            'form_id' => $form->id,
            'type' => 'textarea',
            'name' => 'Message',
        ]);

        $response = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('enquiries.store'), [
                'form_id' => $form->id,
                'questions' => []
            ]);
        $response->assertStatus(422);
        $this->assertArrayHasKey('questions.' . $first_name->id, $response->json('errors'));
        $this->assertArrayHasKey('questions.' . $last_name->id, $response->json('errors'));
        $this->assertArrayHasKey('questions.' . $email->id, $response->json('errors'));
        $this->assertArrayHasKey('questions.' . $phone->id, $response->json('errors'));
        $this->assertArrayNotHasKey('questions.' . $message->id, $response->json('errors'));
    }

    public function testEmailTypeValidation()
    {
        $form = Form::factory()->create();
        $email = FormQuestion::factory()->create([
            'form_id' => $form->id,
            'type' => 'email',
            'name' => 'Email',
            'validation' => 'required|email',
        ]);
        $response = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('enquiries.store'), [
                'form_id' => $form->id,
                'questions' => [
                    $email->id => 'test-me',
                ]
            ]);
        $response->assertStatus(422);
        $this->assertArrayHasKey('questions.' . $email->id, $response->json('errors'));
    }

    public function testSuccessMessageIsReturned()
    {
        $form = Form::factory()->create(['success_message' => 'foo baz bar']);
        $email = FormQuestion::factory()->create([
            'form_id' => $form->id,
            'type' => 'email',
            'name' => 'Email',
            'validation' => 'required|email',
        ]);
        $response = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('enquiries.store'), [
                'form_id' => $form->id,
                'questions' => [
                    $email->id => 'test@example.com',
                ]
            ]);
        $response->assertStatus(200);
        $this->assertArrayHasKey('status', $response->json());
        $this->assertArrayHasKey('message', $response->json());
        $this->assertEquals('foo baz bar', $response->json('message'));
        $this->assertEquals('ok', $response->json('status'));
    }
}
