<?php

namespace Tests\Feature;

use App\Facades\Settings;
use App\HasSeoData;
use App\Seo\OfferSeoDataMap;
use App\Seo\PageSeoDataMap;
use App\Seo\VehicleSeoDataMap;
use Illuminate\Foundation\Testing\DatabaseMigrations;
use Mtc\MercuryDataModels\Page;
use Mtc\MercuryDataModels\SeoData;
use Mtc\MercuryDataModels\SeoDefault;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleMake;
use Mtc\MercuryDataModels\VehicleModel;
use Mtc\MercuryDataModels\VehicleOffer;
use Tests\TestCase;

class SeoDataMapTest extends TestCase
{
    use DatabaseMigrations;
    use HasSeoData;

    protected $tenancy = true;

    public function testVehicleDefaultFields()
    {
        Settings::make([
            'section' => 'general',
            'group' => 'Site Details',
            'name' => 'Site Name',
            'config_key' => 'app.name',
            'type' => 'string',
            'value' => 'my motors'
        ]);

        $make = VehicleMake::factory()->create([
            'name' => 'Audi',
        ]);
        $model = VehicleModel::factory()->create([
            'name' => 'A7',
            'make_id' => $make->id,
        ]);

        $vehicle = Vehicle::factory()->create([
            'title' => 'foo baz bar',
            'manufacture_year' => 2020,
            'make_id' => $make->id,
            'model_id' => $model->id,
            'description' => "<p>Lorem ipsum dolor sit amet, consectetur adipiscing elit. Phasellus a nibh lobortis, "
                . "sodales tortor nec, scelerisque tellus. Cras id neque quis nulla mattis maximus quis sed nisl.</p>"
                . "<p> Integer nec nisi ante. Aenean consectetur placerat libero et euismod. Pellentesque tellus urna,"
                . "fermentum nec enim quis, fermentum pellentesque ante. Proin et tortor et turpis faucibus eleifend.</p>",
        ]);

        $result = (new VehicleSeoDataMap($vehicle))->map('{{TITLE}}');
        $this->assertEquals('foo baz bar', $result);
        $result = (new VehicleSeoDataMap($vehicle))->map('{{YEAR}}');
        $this->assertEquals('2020', $result);
        $result = (new VehicleSeoDataMap($vehicle))->map('{{MAKE}}');
        $this->assertEquals('Audi', $result);
        $result = (new VehicleSeoDataMap($vehicle))->map('{{MODEL}}');
        $this->assertEquals('A7', $result);
        $result = (new VehicleSeoDataMap($vehicle))->map('{{MAKE}} {{MODEL}} {{TITLE}} {{YEAR}} | {{SITE_NAME}}');
        $this->assertEquals('Audi A7 foo baz bar 2020 | my motors', $result);
        $result = (new VehicleSeoDataMap($vehicle))->map('{{CONTENT_EXCERPT}}');
        $this->assertEquals('Lorem ipsum dolor sit amet, consectetur adipiscing elit. Phasellus a nibh lobortis, sodales tortor nec, scelerisque tellus. Cras id neque quis null...', $result);
    }

    public function testOfferDefaultFields()
    {
        Settings::make([
            'section' => 'general',
            'group' => 'Site Details',
            'name' => 'Site Name',
            'config_key' => 'app.name',
            'type' => 'string',
            'value' => 'my motors'
        ]);

        $make = VehicleMake::factory()->create([
            'name' => 'Audi',
        ]);
        $model = VehicleModel::factory()->create([
            'name' => 'A7',
            'make_id' => $make->id,
        ]);

        $offer = VehicleOffer::factory()->create([
            'name' => 'foo baz bar',
            'make_id' => $make->id,
            'model_id' => $model->id,
        ]);

        $result = (new OfferSeoDataMap($offer))->map('{{TITLE}}');
        $this->assertEquals('foo baz bar', $result);
        $result = (new OfferSeoDataMap($offer))->map('{{MAKE}}');
        $this->assertEquals('Audi', $result);
        $result = (new OfferSeoDataMap($offer))->map('{{MODEL}}');
        $this->assertEquals('A7', $result);
        $result = (new OfferSeoDataMap($offer))->map('{{MAKE}} {{MODEL}} {{TITLE}} | {{SITE_NAME}}');
        $this->assertEquals('Audi A7 foo baz bar | my motors', $result);
    }

    public function testPageDefaultFields()
    {
        Settings::make([
            'section' => 'general',
            'group' => 'Site Details',
            'name' => 'Site Name',
            'config_key' => 'app.name',
            'type' => 'string',
            'value' => 'my motors'
        ]);

        $page = Page::factory()->create([
            'title' => 'foo baz bar',
        ]);

        $result = (new PageSeoDataMap($page))->map('{{TITLE}}');
        $this->assertEquals('foo baz bar', $result);
        $result = (new PageSeoDataMap($page))->map('{{TITLE}} | {{SITE_NAME}}');
        $this->assertEquals('foo baz bar | my motors', $result);
    }

    public function testVehicleSeoFromPath()
    {
        $make = VehicleMake::factory()->create([
            'name' => 'Audi',
        ]);
        $model = VehicleModel::factory()->create([
            'name' => 'A7',
            'make_id' => $make->id,
        ]);

        $vehicle = Vehicle::factory()->create([
            'title' => 'foo bar',
            'manufacture_year' => 2020,
            'make_id' => $make->id,
            'model_id' => $model->id,
            'description' => "<p>Lorem ipsum dolor sit amet, consectetur adipiscing elit. Phasellus a nibh lobortis, "
                . "sodales tortor nec, scelerisque tellus. Cras id neque quis nulla mattis maximus quis sed nisl.</p>"
                . "<p> Integer nec nisi ante. Aenean consectetur placerat libero et euismod. Pellentesque tellus urna,"
                . "fermentum nec enim quis, fermentum pellentesque ante. Proin et tortor et turpis faucibus eleifend.</p>",
        ]);

        SeoData::factory()->create([
            'path' => '/vehicle/foo-baz',
            'title' => 'foo baz bar',
            'description' => 'lorem ipsum dolor sit amet',
        ]);

        $result = $this->getSeoData('/vehicle/foo-baz', 'vehicle', new VehicleSeoDataMap($vehicle));

        $this->assertEquals('foo baz bar', $result['title']);
        $this->assertEquals('lorem ipsum dolor sit amet', $result['description']);
    }

    public function testVehicleSeoFromDefaultsWhenPathHasNull()
    {
        Settings::make([
            'section' => 'general',
            'group' => 'Site Details',
            'name' => 'Site Name',
            'config_key' => 'app.name',
            'type' => 'string',
            'value' => 'my motors'
        ]);
        $make = VehicleMake::factory()->create([
            'name' => 'Audi',
        ]);
        $model = VehicleModel::factory()->create([
            'name' => 'A7',
            'make_id' => $make->id,
        ]);

        $vehicle = Vehicle::factory()->create([
            'title' => 'foo baz bar',
            'manufacture_year' => 2020,
            'make_id' => $make->id,
            'model_id' => $model->id,
            'description' => "<p>Lorem ipsum dolor sit amet, consectetur adipiscing elit. Phasellus a nibh lobortis, "
                . "sodales tortor nec, scelerisque tellus. Cras id neque quis nulla mattis maximus quis sed nisl.</p>"
                . "<p> Integer nec nisi ante. Aenean consectetur placerat libero et euismod. Pellentesque tellus urna,"
                . "fermentum nec enim quis, fermentum pellentesque ante. Proin et tortor et turpis faucibus eleifend.</p>",
        ]);

        SeoDefault::factory()->create([
            'section' => 'vehicle',
            'title' => '{{TITLE}} | {{SITE_NAME}}'
        ]);

        SeoData::factory()->create([
            'path' => '/vehicle/foo-baz',
            'title' => null,
            'description' => 'lorem ipsum dolor sit amet',
        ]);

        $result = $this->getSeoData('/vehicle/foo-baz', 'vehicle', new VehicleSeoDataMap($vehicle));

        $this->assertEquals('foo baz bar | my motors', $result['title']);
        $this->assertEquals('lorem ipsum dolor sit amet', $result['description']);
    }
}
