<?php

namespace Tests\Feature;

use App\Facades\Settings;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\App;
use Mtc\Filter\Contracts\FilterInstance;
use Mtc\MercuryDataModels\BodyStyleType;
use Mtc\MercuryDataModels\Dealership;
use Mtc\MercuryDataModels\FuelType;
use Mtc\MercuryDataModels\TransmissionType;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleMake;
use Mtc\MercuryDataModels\VehicleModel;
use Tests\FilterSettingSeed;
use Tests\TestCase;

class VehicleFiltersTest extends TestCase
{
    use RefreshDatabase;

    protected $tenancy = true;

    public function testFilterByMake()
    {
        $this->seed(FilterSettingSeed::class);
        $make = VehicleMake::factory()->create();
        $make2 = VehicleMake::factory()->create();
        Vehicle::factory(5)->create(['make_id' => $make->id]);
        Vehicle::factory(3)->create(['make_id' => $make2->id]);
        Vehicle::factory(10)->create(['make_id' => 100]);

        // No makes specified
        $response = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'));
        $response->assertStatus(200);
        $this->assertEquals(18, $response->json('results.total'));

        // First Make
        $response2 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'make',
                        'value' => $make->id,
                    ],
                ]
            ]);
        $response2->assertStatus(200);
        $this->assertEquals(5, $response2->json('results.total'));

        // Second make
        $response3 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'make',
                        'value' => $make2->id,
                    ],
                ]
            ]);
        $response3->assertStatus(200);
        $this->assertEquals(3, $response3->json('results.total'));

        // Both Makes
        $response4 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'make',
                        'value' => $make->id,
                    ],
                    [
                        'type' => 'make',
                        'value' => $make2->id,
                    ],
                ]
            ]);
        $response4->assertStatus(200);
        $this->assertEquals(8, $response4->json('results.total'));
    }

    public function testFilterByModel()
    {
        $this->seed(FilterSettingSeed::class);
        $model = VehicleModel::factory()->create();
        $model2 = VehicleModel::factory()->create();
        Vehicle::factory(5)->create(['model_id' => $model->id]);
        Vehicle::factory(3)->create(['model_id' => $model2->id]);
        Vehicle::factory(10)->create(['model_id' => 100]);

        // No  specified
        $response = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'));
        $response->assertStatus(200);
        $this->assertEquals(18, $response->json('results.total'));

        // First
        $response2 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'model',
                        'value' => $model->id,
                    ],
                ]
            ]);
        $response2->assertStatus(200);
        $this->assertEquals(5, $response2->json('results.total'));

        // Second
        $response3 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'model',
                        'value' => $model2->id,
                    ],
                ]
            ]);
        $response3->assertStatus(200);
        $this->assertEquals(3, $response3->json('results.total'));

        // Both
        $response4 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'model',
                        'value' => $model->id,
                    ],
                    [
                        'type' => 'model',
                        'value' => $model2->id,
                    ],
                ]
            ]);
        $response4->assertStatus(200);
        $this->assertEquals(8, $response4->json('results.total'));
    }

    public function testFilterByAge()
    {
        $this->seed(FilterSettingSeed::class);
        Vehicle::factory(5)->create(['manufacture_year' => 2021]);
        Vehicle::factory(3)->create(['manufacture_year' => 2015]);
        Vehicle::factory(10)->create(['manufacture_year' => 2009]);

        // No makes specified
        $response = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'));
        $response->assertStatus(200);
        $this->assertEquals(18, $response->json('results.total'));

        // First
        $response2 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'age',
                        'value' => [
                            'min' => 2020,
                            'max' => 2022,
                        ],
                    ],
                ]
            ]);
        $response2->assertStatus(200);
        $this->assertEquals(5, $response2->json('results.total'));

        // Second
        $response3 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'age',
                        'value' => [
                            'min' => 2015,
                            'max' => 2020,
                        ],
                    ],
                ]
            ]);
        $response3->assertStatus(200);
        $this->assertEquals(3, $response3->json('results.total'));

        // Second
        $response3 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'age',
                        'value' => [
                            'min' => 2015,
                            'max' => 2022,
                        ],
                    ],
                ]
            ]);
        $response3->assertStatus(200);
        $this->assertEquals(8, $response3->json('results.total'));
    }

    public function testFilterByBodyStyle()
    {
        $this->seed(FilterSettingSeed::class);
        $body_style = BodyStyleType::factory()->create();
        $body_style2 = BodyStyleType::factory()->create();
        Vehicle::factory(5)->create(['body_style_id' => $body_style->id]);
        Vehicle::factory(3)->create(['body_style_id' => $body_style2->id]);
        Vehicle::factory(10)->create(['body_style_id' => 100]);

        // No specified
        $response = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'));
        $response->assertStatus(200);
        $this->assertEquals(18, $response->json('results.total'));

        // First
        $response2 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'body_type',
                        'value' => $body_style->id,
                    ],
                ]
            ]);
        $response2->assertStatus(200);
        $this->assertEquals(5, $response2->json('results.total'));

        // Second
        $response3 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'body_type',
                        'value' => $body_style2->id,
                    ],
                ]
            ]);
        $response3->assertStatus(200);
        $this->assertEquals(3, $response3->json('results.total'));

        // Both
        $response4 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'body_type',
                        'value' => $body_style->id,
                    ],
                    [
                        'type' => 'body_type',
                        'value' => $body_style2->id,
                    ],
                ]
            ]);
        $response4->assertStatus(200);
        $this->assertEquals(8, $response4->json('results.total'));
    }

    public function testFilterByColour()
    {
        $this->seed(FilterSettingSeed::class);
        Vehicle::factory(5)->create(['colour' => 'red']);
        Vehicle::factory(3)->create(['colour' => 'black']);
        Vehicle::factory(10)->create(['colour' => 'silver']);

        // No makes specified
        $response = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'));
        $response->assertStatus(200);
        $this->assertEquals(18, $response->json('results.total'));

        // First
        $response2 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'colour',
                        'value' => 'red',
                    ],
                ]
            ]);
        $response2->assertStatus(200);
        $this->assertEquals(5, $response2->json('results.total'));

        // Second
        $response3 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'colour',
                        'value' => 'black',
                    ],
                ]
            ]);
        $response3->assertStatus(200);
        $this->assertEquals(3, $response3->json('results.total'));

        // Both
        $response4 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'colour',
                        'value' => 'red',
                    ],
                    [
                        'type' => 'colour',
                        'value' => 'black',
                    ],
                ]
            ]);
        $response4->assertStatus(200);
        $this->assertEquals(8, $response4->json('results.total'));
    }


    public function testFilterByDoorCount()
    {
        $this->seed(FilterSettingSeed::class);
        Vehicle::factory(5)->create(['door_count' => 4]);
        Vehicle::factory(3)->create(['door_count' => 5]);
        Vehicle::factory(10)->create(['door_count' => 3]);

        // No makes specified
        $response = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'));
        $response->assertStatus(200);
        $this->assertEquals(18, $response->json('results.total'));

        // First
        $response2 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'door_count',
                        'value' => 4,
                    ],
                ]
            ]);
        $response2->assertStatus(200);
        $this->assertEquals(5, $response2->json('results.total'));

        // Second
        $response3 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'door_count',
                        'value' => 5,
                    ],
                ]
            ]);
        $response3->assertStatus(200);
        $this->assertEquals(3, $response3->json('results.total'));

        // Both
        $response4 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'door_count',
                        'value' => 4,
                    ],
                    [
                        'type' => 'door_count',
                        'value' => 5,
                    ],
                ]
            ]);
        $response4->assertStatus(200);
        $this->assertEquals(8, $response4->json('results.total'));
    }

    public function testFilterByEngineSize()
    {
        $this->seed(FilterSettingSeed::class);
        Vehicle::factory(5)->create(['engine_size_cc' => 1400]);
        Vehicle::factory(3)->create(['engine_size_cc' => 1999]);
        Vehicle::factory(10)->create(['engine_size_cc' => 2500]);

        // No makes specified
        $response = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'));
        $response->assertStatus(200);
        $this->assertEquals(18, $response->json('results.total'));

        // First
        $response2 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'engine_size',
                        'value' => [
                            'min' => 1000,
                            'max' => 1500,
                        ],
                    ],
                ]
            ]);
        $response2->assertStatus(200);
        $this->assertEquals(5, $response2->json('results.total'));

        // Second
        $response3 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'engine_size',
                        'value' => [
                            'min' => 1500,
                            'max' => 2000,
                        ],
                    ],
                ]
            ]);
        $response3->assertStatus(200);
        $this->assertEquals(3, $response3->json('results.total'));

        // Second
        $response3 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'engine_size',
                        'value' => [
                            'min' => 1000,
                            'max' => 2000,
                        ],
                    ],
                ]
            ]);
        $response3->assertStatus(200);
        $this->assertEquals(8, $response3->json('results.total'));
    }

    public function testFilterByFuelType()
    {
        $this->seed(FilterSettingSeed::class);
        $fuel_type = FuelType::factory()->create();
        $fuel_type2 = FuelType::factory()->create();
        Vehicle::factory(5)->create(['fuel_type_id' => $fuel_type->id]);
        Vehicle::factory(3)->create(['fuel_type_id' => $fuel_type2->id]);
        Vehicle::factory(10)->create(['fuel_type_id' => 100]);

        // No makes specified
        $response = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'));
        $response->assertStatus(200);
        $this->assertEquals(18, $response->json('results.total'));

        // First Make
        $response2 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'fuel_type',
                        'value' => $fuel_type->id,
                    ],
                ]
            ]);
        $response2->assertStatus(200);
        $this->assertEquals(5, $response2->json('results.total'));

        // Second
        $response3 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'fuel_type',
                        'value' => $fuel_type2->id,
                    ],
                ]
            ]);
        $response3->assertStatus(200);
        $this->assertEquals(3, $response3->json('results.total'));

        // Both
        $response4 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'fuel_type',
                        'value' => $fuel_type->id,
                    ],
                    [
                        'type' => 'fuel_type',
                        'value' => $fuel_type2->id,
                    ],
                ]
            ]);
        $response4->assertStatus(200);
        $this->assertEquals(8, $response4->json('results.total'));
    }

    public function testFilterByLocation()
    {
        $this->seed(FilterSettingSeed::class);
        $location = Dealership::factory()->create();
        $location2 = Dealership::factory()->create();
        Vehicle::factory(5)->create(['dealership_id' => $location->id]);
        Vehicle::factory(3)->create(['dealership_id' => $location2->id]);
        Vehicle::factory(10)->create(['dealership_id' => 100]);

        // No makes specified
        $response = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'));
        $response->assertStatus(200);
        $this->assertEquals(18, $response->json('results.total'));

        // First
        $response2 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'location',
                        'value' => $location->id,
                    ],
                ]
            ]);
        $response2->assertStatus(200);
        $this->assertEquals(5, $response2->json('results.total'));

        // Second
        $response3 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'location',
                        'value' => $location2->id,
                    ],
                ]
            ]);
        $response3->assertStatus(200);
        $this->assertEquals(3, $response3->json('results.total'));

        // Both
        $response4 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'location',
                        'value' => $location->id,
                    ],
                    [
                        'type' => 'location',
                        'value' => $location2->id,
                    ],
                ]
            ]);
        $response4->assertStatus(200);
        $this->assertEquals(8, $response4->json('results.total'));
    }

    public function testFilterByMileage()
    {
        $this->seed(FilterSettingSeed::class);
        Settings::make([
            'section' => 'automotive',
            'group' => 'General',
            'name' => 'Distance measurement',
            'config_key' => 'automotive.distance_measurement',
            'type' => 'string',
            'value' => 'mi',
        ]);
        Vehicle::factory(5)->create(['odometer_mi' => 60000]);
        Vehicle::factory(3)->create(['odometer_mi' => 45000]);
        Vehicle::factory()->create(['odometer_mi' => 48000]);
        Vehicle::factory(10)->create(['odometer_mi' => 25000]);

        // No makes specified
        $response = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'));
        $response->assertStatus(200);
        $this->assertEquals(19, $response->json('results.total'));

        // First
        $response2 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'mileage',
                        'value' => [
                            'min' => 50000,
                            'max' => 75000,
                        ],
                    ],
                ]
            ]);
        $response2->assertStatus(200);
        $this->assertEquals(5, $response2->json('results.total'));

        // Second
        $response3 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'mileage',
                        'value' => [
                            'min' => 40000,
                            'max' => 50000,
                        ],
                    ],
                ]
            ]);
        $response3->assertStatus(200);
        $this->assertEquals(4, $response3->json('results.total'));

        // Second
        $response3 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'mileage',
                        'value' => [
                            'min' => 40000,
                            'max' => 46000,
                        ],
                    ],
                ]
            ]);
        $response3->assertStatus(200);
        $this->assertEquals(3, $response3->json('results.total'));
    }

    public function testFilterByPrice()
    {
        $this->seed(FilterSettingSeed::class);
        Vehicle::factory(5)->create(['price' => 10200, 'monthly_price' => 250]);
        Vehicle::factory(3)->create(['price' => 8500, 'monthly_price' => 170]);
        Vehicle::factory(10)->create(['price' => 5000, 'monthly_price' => 130]);

        // No makes specified
        $response = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'));
        $response->assertStatus(200);
        $this->assertEquals(18, $response->json('results.total'));

        // First
        $response2 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'price',
                        'value' => [
                            'payment_type' => 'full',
                            'min' => 10000,
                            'max' => 12500,
                        ],
                    ],
                ]
            ]);
        $response2->assertStatus(200);
        $this->assertEquals(5, $response2->json('results.total'));

        // Second
        $response3 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'price',
                        'value' => [
                            'payment_type' => 'full',
                            'min' => 7500,
                            'max' => 10000,
                        ],
                    ],
                ]
            ]);
        $response3->assertStatus(200);
        $this->assertEquals(3, $response3->json('results.total'));

        $response4 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'price',
                        'value' => [
                            'payment_type' => 'monthly',
                            'min' => 250,
                            'max' => 300,
                        ],
                    ],
                ]
            ]);
        $response4->assertStatus(200);
        $this->assertEquals(5, $response4->json('results.total'));

        $response4 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'price',
                        'value' => [
                            'payment_type' => 'monthly',
                            'min' => 150,
                            'max' => 200,
                        ],
                    ],
                ]
            ]);
        $response4->assertStatus(200);
        $this->assertEquals(3, $response4->json('results.total'));
    }

    public function testFilterBySeatCount()
    {
        $this->seed(FilterSettingSeed::class);
        Vehicle::factory(5)->create(['seats' => 4]);
        Vehicle::factory(3)->create(['seats' => 2]);
        Vehicle::factory(10)->create(['seats' => 5]);

        // No makes specified
        $response = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'));
        $response->assertStatus(200);
        $this->assertEquals(18, $response->json('results.total'));

        // First
        $response2 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'seats',
                        'value' => 4,
                    ],
                ]
            ]);
        $response2->assertStatus(200);
        $this->assertEquals(5, $response2->json('results.total'));

        // Second
        $response3 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'seats',
                        'value' => 2,
                    ],
                ]
            ]);
        $response3->assertStatus(200);
        $this->assertEquals(3, $response3->json('results.total'));

        // Both
        $response4 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'seats',
                        'value' => 4,
                    ],
                    [
                        'type' => 'seats',
                        'value' => 2,
                    ],
                ]
            ]);
        $response4->assertStatus(200);
        $this->assertEquals(8, $response4->json('results.total'));
    }

    public function testFilterByTransmission()
    {
        $this->seed(FilterSettingSeed::class);
        $transmission = TransmissionType::factory()->create();
        $transmission2 = TransmissionType::factory()->create();
        Vehicle::factory(5)->create(['transmission_id' => $transmission->id]);
        Vehicle::factory(3)->create(['transmission_id' => $transmission2->id]);
        Vehicle::factory(10)->create(['transmission_id' => 100]);

        // No makes specified
        $response = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'));
        $response->assertStatus(200);
        $this->assertEquals(18, $response->json('results.total'));

        // First
        $response2 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'transmission',
                        'value' => $transmission->id,
                    ],
                ]
            ]);
        $response2->assertStatus(200);
        $this->assertEquals(5, $response2->json('results.total'));

        // Second
        $response3 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'transmission',
                        'value' => $transmission2->id,
                    ],
                ]
            ]);
        $response3->assertStatus(200);
        $this->assertEquals(3, $response3->json('results.total'));

        // Both
        $response4 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'transmission',
                        'value' => $transmission->id,
                    ],
                    [
                        'type' => 'transmission',
                        'value' => $transmission2->id,
                    ],
                ]
            ]);
        $response4->assertStatus(200);
        $this->assertEquals(8, $response4->json('results.total'));
    }

    public function testFilterBySearchTerm()
    {
        $this->markTestSkipped('Unable to test fulltext search in sqlite');
    }

    public function testFilterByMultipleFilters()
    {
        $this->seed(FilterSettingSeed::class);
        $transmission = TransmissionType::factory()->create();
        $transmission2 = TransmissionType::factory()->create();
        Vehicle::factory(3)->create([
            'transmission_id' => $transmission->id,
            'colour' => 'blue',
        ]);
        Vehicle::factory(4)->create([
            'transmission_id' => $transmission->id,
            'colour' => 'red',
        ]);
        Vehicle::factory(5)->create([
            'transmission_id' => $transmission2->id,
            'colour' => 'blue',
        ]);
        Vehicle::factory(6)->create([
            'transmission_id' => $transmission2->id,
            'colour' => 'red',
        ]);
        Vehicle::factory(10)->create(['transmission_id' => 100]);

        // No makes specified
        $response = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'));
        $response->assertStatus(200);
        $this->assertEquals(28, $response->json('results.total'));

        // First
        $response2 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'transmission',
                        'value' => $transmission->id,
                    ],
                    [
                        'type' => 'colour',
                        'value' => 'red',
                    ],
                ]
            ]);
        $response2->assertStatus(200);
        $this->assertEquals(4, $response2->json('results.total'));

        // Second
        $response3 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'transmission',
                        'value' => $transmission->id,
                    ],
                    [
                        'type' => 'colour',
                        'value' => 'blue',
                    ],
                ]
            ]);
        $response3->assertStatus(200);
        $this->assertEquals(3, $response3->json('results.total'));

        // Both
        $response4 = $this->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'transmission',
                        'value' => $transmission->id,
                    ],
                    [
                        'type' => 'transmission',
                        'value' => $transmission2->id,
                    ],
                    [
                        'type' => 'colour',
                        'value' => 'red',
                    ],
                ]
            ]);
        $response4->assertStatus(200);
        $this->assertEquals(10, $response4->json('results.total'));
    }

    public function testFiltersAreEnabledBySettings()
    {
        $this->seed(FilterSettingSeed::class);
        $filters = App::make(FilterInstance::class)->getFilters();
        $this->assertEquals(14, $filters->count());

        Settings::update('automotive.vehicle-filters.colour', false);
        Settings::update('automotive.vehicle-filters.fuel_type', false);
        $filters = App::make(FilterInstance::class)->getFilters();
        $this->assertEquals(12, $filters->count());
        $this->assertArrayNotHasKey('colour', $filters);
        $this->assertArrayNotHasKey('fuel_type', $filters);
    }
}
