<?php

namespace App\Http\Controllers;

use App\ConfiguratorRepository;
use App\Facades\Settings;
use App\Http\Requests\ConfiguratorRequest;
use App\Http\Requests\StoreConfiguratorRequest;
use App\Http\Resources\ConfiguratorList;
use App\Http\Resources\ConfiguratorView;
use App\Http\Resources\FormViewResource;
use Illuminate\Http\Response;
use Mtc\MercuryDataModels\CarConfiguratorModel;
use Mtc\MercuryDataModels\CustomerCarConfiguration;
use Mtc\MercuryDataModels\Form;

class ConfiguratorController extends Controller
{
    public function index()
    {
        return new ConfiguratorList(CarConfiguratorModel::query()->where('active', 1)->get());
    }
    public function show(ConfiguratorRequest $request, string $slug): ConfiguratorView
    {
        $model = CarConfiguratorModel::query()
            ->where('slug', $slug)
            ->where('active', 1)
            ->firstOrFail();
        return new ConfiguratorView($model);
    }

    public function store(
        StoreConfiguratorRequest $request,
        CarConfiguratorModel $model,
        ConfiguratorRepository $repository
    ): Response {
        $result = $repository->storeLead($request, $model);
        if ($result) {
            return response([
                'id' => $result->id,
                'uuid' => $result->uuid,
                'actions' => $this->actions($result),
                'message' => 'Configuration saved successfully'
            ], 201);
        }

        return response(['error' => 'Failed to store configuration'], 400);
    }

    public function view(ConfiguratorRequest $request, string $uuid): array
    {
        $configuration = CustomerCarConfiguration::query()
            ->with('model')
            ->where('uuid', $uuid)
            ->firstOrFail();
        $configuration->increment('views');
        if (!$configuration->model) {
            abort(404, 'Configuration model not found');
        }
        return [
            'customer' => $configuration->only([
                'uuid',
                'email',
                'name',
                'contact_number',
            ]),
            'configurator' => (new ConfiguratorView($configuration->model))
                ->setSelectedSections($configuration->sections)
        ];
    }

    private function actions(CustomerCarConfiguration $result)
    {
        $enquireForm = $testDriveForm = null;
        if (Settings::get('configurator-enquiry-form')) {
            $enquiry = Form::query()
                ->with('sections.questions')
                ->where('is_active', 1)
                ->find(Settings::get('configurator-enquiry-form'));
            if ($enquiry) {
                $enquireForm = (new FormViewResource($enquiry))->setValues([
                    'car_configuration_id' => $result->id,
                ]);
            }
        }
        if (Settings::get('configurator-test-drive-form')) {
            $testDrive = Form::query()
                ->with('sections.questions')
                ->where('is_active', 1)
                ->find(Settings::get('configurator-test-drive-form'));
            if ($testDrive) {
                $testDriveForm = (new FormViewResource($testDrive))->setValues([
                    'car_configuration_id' => $result->id,
                ]);
            }
        }
        return [
            'enquire' => $enquireForm,
            'test_drive' => $testDriveForm,
        ];
    }
}
