<?php

namespace Tests\Feature;

use App\Facades\Settings;
use Mtc\MercuryDataModels\Finance\Contracts\FinanceRequestData;
use Mtc\MercuryDataModels\Finance\FinanceServiceHelper;
use Carbon\Carbon;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Http\Client\Request;
use Illuminate\Support\Facades\Http;
use Tests\Data\FinanceSettingSeeder;
use Tests\TestCase;

class EvolutionServiceTest extends TestCase
{
    use RefreshDatabase;

    protected $tenancy = true;

    public function testGoodRequestCalculation()
    {
        Http::fake([
            "https://www.evolutionfunding.com/api/v3/batch" => Http::response($this->response()),
            "*" => Http::response([], 400),
        ]);

        $this->seed(FinanceSettingSeeder::class);
        Settings::update('finance-evolution-enabled', true);
        Settings::update('finance-evolution-client_id', 'foo-baz-bar');
        Settings::update('finance-evolution-client_password', 'lorem');
        $provider = FinanceServiceHelper::initializeForSite();

        $result = $provider->calculate(new FinanceRequestData(
            uuid: 'abc123',
            registration_number: 'SD12 ABC',
            cap_id: 1231122,
            dealer_id: 'abc123',
            engine_size: 1999,
            condition: 'new',
            registration_date: Carbon::create(2020, 1, 1),
            term: 60,
            mileage: 100000,
            annual_mileage: 13000,
            price: 23456,
            deposit: 3000,
            credit_rating: 'excellent',
        ));

        Http::assertSent(function (Request $request) {
            return $request->hasHeader('id', 'foo-baz-bar')
                && $request->hasHeader('password', 'lorem')
                && $request->url() === 'https://www.evolutionfunding.com/api/v3/batch'
                && $request['term'] === 60
                && $request['current_miles'] === 100000
                && $request['cprice'] == 23456;
        });

        $this->assertEquals(2, $result->count());
        $finance = $result->filter(fn($quote) => ($quote != null && $quote->finance_type->value == 'PCP'))->first();
        $this->assertEquals(123, $finance->monthly_price);
        $this->assertEquals(60, $finance->number_of_payments);
        $this->assertEquals(13, $finance->apr);
        $this->assertEquals(3000, $finance->total_deposit);
        $this->assertEquals(23456, $finance->cash_price);
        $this->assertEquals(0, $finance->dealer_deposit);
        $this->assertEquals(3000, $finance->customer_deposit);
        $this->assertEquals(24456, $finance->payable_amount);
        $this->assertEquals(0, $finance->option_to_purchase_fee);
        $this->assertEquals(0, $finance->documentation_fee);
    }

    private function response(): array
    {
        return [
            'data' => [
                'pcp' => [
                    23456 => [
                        60 => [
                            3000 => [
                                13000 => [
                                    [
                                        'finance_type' => 'pcp',
                                        'regular' => 123,
                                        'term' => 60,
                                        'deposit' => 3000,
                                        'apr' => 13,
                                        'first' => 132,
                                        'final' => 1200,
                                        'rv_interest' => 12,
                                        'product_name' => 'example PCP',
                                        'charges' => 1000,
                                        'total' => 24456,
                                    ],
                                ]
                            ]
                        ]
                    ]
                ],
            ]
        ];
    }
}
