<?php

namespace App\Http\Resources;

use App\Traits\HasPlaceholderImage;
use App\Traits\MediaHasSizes;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\Resources\Json\JsonResource;
use Illuminate\Support\Collection;
use Mtc\MercuryDataModels\NewCar;
use Mtc\MercuryDataModels\VehicleOffer;

class NewCarListResource extends JsonResource
{
    use HasPlaceholderImage;
    use MediaHasSizes;

    public static $wrap = '';

    private Collection $placeholder;
    /**
     * Transform the resource into an array.
     *
     * @param \Illuminate\Http\Request $request
     * @return array|\Illuminate\Contracts\Support\Arrayable|\JsonSerializable
     */
    public function toArray($request)
    {
        if ($this->resource instanceof LengthAwarePaginator) {
            $this->resource->through(fn(NewCar $car) => $this->map($car));
        } else {
            $this->resource = $this->resource->map(fn(NewCar $car) => $this->map($car));
        }
        return parent::toArray($request);
    }

    protected function map(NewCar $car): array
    {
        $fuelTypes = null;
        if ($car->fuelTypes->count() == 1) {
            $fuelTypes = $car->fuelTypes->first()->name;
        } elseif ($car->fuelTypes->count() > 1) {
            $fuelTypes = $car->fuelTypes->pluck('name');
        }

        $previewImage = $car->mediaUses
            ->filter(fn ($mediaUse) => $mediaUse->flags['listing'] ?? null)
            ->first();

        return [
            'slug' => $car->slug,
            'name' => $car->name,
            'excerpt' => $car->excerpt,
            'derivative' => $car->derivative,
            'make' => $car->make->name,
            'model' => $car->model->name,
            'image' => $this->getPreviewImage($car, 'tile', 'listing'),
            'image_sizes' => !empty($previewImage)
                ? $this->allSizesForUse($car, $previewImage)
                : [],
            'hero_image' => $this->getPreviewImage($car, 'tile', 'hero'),
            'price' => $car->price,
            'monthly_price' => $car->monthly_price,
            'section_url' => $car->make->filterIndex->slug . '/' . $car->model->filterIndex->slug,
            'fuel_type' => $fuelTypes,
            'colour' => $car->colour,
            'fuel_types' => $car->getAvailableFuelTypes(),
            'from_price' => $this->getFromPriceFromOffers($car),
        ];
    }

    public function getPreviewImage(Model $model, string $dimensions, string $type): string
    {
        $listingImage = $model->mediaUses
            ->filter(fn ($mediaUse) => $mediaUse->flags[$type] ?? null)
            ->first();

        return $listingImage?->getUrl($dimensions) ?? $this->getPlaceholder($model, $dimensions);
    }

    protected function largeImageDimensions(): string
    {
        return 'tile';
    }

    protected function mediumImageDimensions(): string
    {
        return $this->largeImageDimensions();
    }

    protected function thumbnailImageDimensions(): string
    {
        return $this->largeImageDimensions();
    }

    protected function getFromPriceFromOffers(NewCar $car): ?float
    {
        return VehicleOffer::query()
            ->where('make_id', '=', $car->make_id)
            ->where('model_id', '=', $car->model_id)
            ->orderBy('price')
            ->pluck('price')
            ->first();
    }
}
