<?php

namespace App;

use App\Facades\Settings;
use App\Metrics\EnquiryCount;
use Carbon\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Storage;
use Mtc\MercuryDataModels\Repositories\FeatureRepository;
use Mtc\VehicleReservations\Reservation;
use Spatie\Analytics\Period;

class GraphService
{
    private Collection $data;

    public function __construct(private readonly FeatureRepository $features)
    {
    }

    public function handle(array $metrics, int $dateRange = 30)
    {
        $this->prepareDateRange(Period::days($dateRange));
        if ($this->requiresAnalytics($metrics) && !empty(Settings::get('seo-view_id'))) {
            $this->addCollectionData($this->runAnalytics($metrics, $dateRange));
        }
        $remainingMetrics = array_diff($metrics, Analytics::$allowedMetrics);
        if (!empty($remainingMetrics)) {
            collect($remainingMetrics)
                ->each(function ($metric) use ($dateRange) {
                    $metric_data = $this->runMetric($metric, $dateRange);
                    if ($metric_data->isNotEmpty()) {
                        $this->addToData($metric_data, $metric);
                    }
                });
        }

        return $this->data;
    }

    public function getLabels()
    {
        return $this->data
            ->sortBy('date')
            ->pluck('date')
            ->map(fn($date) => Carbon::parse($date)->format('jS M'));
    }

    private function requiresAnalytics(array $metrics): bool
    {
        return !empty(array_intersect(Analytics::$allowedMetrics, $metrics));
    }

    private function runAnalytics(array $metrics, int $dateRange)
    {
        $this->prepareAnalytics();
        return App::make(Analytics::class)
            ->setPropertyId(Settings::get('seo-view_id'))
            ->runReport(Period::days($dateRange), array_values(array_intersect(Analytics::$allowedMetrics, $metrics)));
    }

    private function prepareAnalytics(): void
    {
        Config::set('analytics.view_id', Settings::get('seo-view_id'));
        $filename = 'file-storage/' . config('analytics.service_account_credentials_json');
        Config::set(
            'analytics.service_account_credentials_json',
            json_decode(Storage::disk(Config::get('filesystems.default'))->get($filename), true)
        );
    }

    private function runMetric(string $metric, int $dateRange)
    {
        try {
            return match ($metric) {
                'enquiries' => $this->ensureFilledArray(
                    (new EnquiryCount())->get(Period::days($dateRange)),
                    $dateRange
                ),
                'reservations' => $this->ensureFilledArray($this->getReservationsForDateRange($dateRange), $dateRange),
                default => collect([]),
            };
        } catch (\Exception $exception) {
            return collect([]);
        }
    }

    private function prepareDateRange(Period $period): void
    {
        $this->data = collect();
        $from = Carbon::parse($period->startDate);
        $to = Carbon::parse($period->endDate);
        $this->data->put(
            $from->format('Ymd'),
            [
                'date' => $from->format('Ymd')
            ],
        );
        do {
            $from->addDay();
            $this->data->put(
                $from->format('Ymd'),
                [
                    'date' => $from->format('Ymd')
                ],
            );
        } while ($from->lt($to));
    }

    private function addToData(Collection $data, string $metric = null)
    {
        $data->each(function ($value, $date) use ($metric) {
            $dateValue = Carbon::parse($date)->format('Ymd');
            $this->data->put($dateValue, array_merge(
                [$metric => $value],
                $this->data[$dateValue] ?? []
            ));
        });
    }

    private function addCollectionData(Collection $data)
    {
        $data->each(function (Collection $value) {
            $dateValue = Carbon::parse($value['date'])->format('Ymd');
            $this->data->put($dateValue, array_merge(
                $value->except(['date'])->toArray(),
                $this->data[$dateValue] ?? []
            ));
        });
    }

    private function ensureFilledArray(Collection $data, int $dateRange): Collection
    {
        return collect(range(0, $dateRange))
            ->map(fn($daysInPast) => Carbon::today()->subDays($daysInPast))
            ->keyBy(fn(Carbon $date) => $date->format('Y-m-d'))
            ->map(fn(Carbon $date) => $data[$date->format('Y-m-d')] ?? 0);
    }

    private function getReservationsForDateRange(int $dateRange)
    {
        if (!$this->features->isEnabled('reservations')) {
            throw new \Exception('Reservations not enabled');
        }
        return Reservation::query()
            ->whereNotNull('confirmed_at')
            ->where('confirmed_at', '>', Carbon::now()->startOfDay()->subDays($dateRange))
            ->where('confirmed_at', '<=', Carbon::now()->endOfDay())
            ->get()
            ->groupBy(fn($reservation) => $reservation->confirmed_at->format('Y-m-d'))
            ->map(fn($reservations_in_date) => $reservations_in_date->count());
    }
}
