<?php

namespace App;

use Carbon\Carbon;
use Google\Analytics\Data\V1beta\BetaAnalyticsDataClient;
use Google\Analytics\Data\V1beta\DateRange;
use Google\Analytics\Data\V1beta\Dimension;
use Google\Analytics\Data\V1beta\Metric;
use Google\ApiCore\ValidationException;
use Illuminate\Contracts\Cache\Repository;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Traits\Macroable;
use Spatie\Analytics\Period;

class Analytics
{
    use Macroable;

    public static array $allowedMetrics = [
        'sessions',
        'totalUsers',
        'conversions',
        'engagementRate',
        'eventCount',
    ];

    private string $propertyId;
    protected int $cacheLifeTimeInMinutes = 30;

    /**
     * @throws ValidationException
     */
    public function __construct(
        protected Repository $cache,
    ) {
        $this->analytics = new BetaAnalyticsDataClient([
            'credentials' => Config::get('analytics.service_account_credentials_json'),
        ]);
    }

    public function setPropertyId(string $propertyId): self
    {
        $this->propertyId = $propertyId;

        return $this;
    }

    public function getPropertyId()
    {
        return $this->propertyId;
    }

    /**
     * Run report for period with metrics across a dimension
     * Supported metrics / dimensions: https://developers.google.com/analytics/devguides/reporting/data/v1/api-schema
     *
     * @param Period $period
     * @param array $metrics
     * @param string $dimension
     * @return Collection|null
     */
    public function runReport(
        Period $period,
        array $metrics = ['sessions', 'totalUsers'],
        string $dimension = 'date'
    ): Collection|null {
        $cacheName = $this->determineCacheName(func_get_args());

        if ($this->cacheLifeTimeInMinutes === 0) {
            Cache::forget($cacheName);
        }

        return Cache::remember(
            $cacheName,
            Carbon::now()->addMinutes($this->cacheLifeTimeInMinutes),
            function () use ($period, $metrics, $dimension) {
                $data = collect();
                $response = $this->analytics->runReport([
                    'property' => 'properties/' . $this->propertyId,
                    'dateRanges' => [
                        new DateRange([
                            'start_date' => $period->startDate->format('Y-m-d'),
                            'end_date' => $period->endDate->format('Y-m-d'),
                        ]),
                    ],
                    'dimensions' => [
                        new Dimension(['name' => $dimension]),
                    ],
                    'metrics' => collect($metrics)
                        ->map(fn($metric) => new Metric(['name' => $metric]))
                        ->toArray(),
                ]);

                foreach ($response->getRows() as $row) {
                    $data[] = collect($row->getMetricValues())
                        ->keyBy(fn($value, $index) => $metrics[$index])
                        ->map(fn($value) => $value->getValue())
                        ->put($dimension, $row->getDimensionValues()[0]->getValue());
                }
                return collect($data)->sort(fn($entry) => $entry[$dimension]);
            }
        );
    }

    /**
     * Determine the cache name for the set of query properties given.
     */
    protected function determineCacheName(array $properties): string
    {
        return 'laravel-analytics.' . md5(serialize($properties));
    }
}
