<?php

namespace Mtc\BankTransfer;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Checkout\Contracts\PayableContract;
use Mtc\Checkout\Contracts\PaymentGateway;
use Mtc\Checkout\PaymentForm;

/**
 * Bank Transfer Payment Gateway
 *
 * @package  Mtc\BankTransfer
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class BankTransfer implements PaymentGateway
{
    /**
     * Check if the gateway is available for use on this payment.
     *
     * @param InvoiceRepositoryContract $invoice
     * @param PayableContract $payable
     * @return bool
     */
    public function isApplicable(InvoiceRepositoryContract $invoice, $payable): bool
    {
        return App::make(config('bank_transfer.applicable_check_class'))->handle($invoice, $payable);
    }

    /**
     * Render the payment template.
     *
     * @param InvoiceRepositoryContract $invoice
     * @param PayableContract $payable
     * @return string
     */
    public function getPaymentForm(InvoiceRepositoryContract $invoice): PaymentForm
    {
        $invoice = template('bank_transfer/payment_form.twig', [
            'invoice' => $invoice
        ]);
        return new PaymentForm($invoice, 'template', [
            'name' => __('bank_transfer::bank_transfer.payment_option_name')
        ]);
    }

    /**
     * Charge payment on invoice
     *
     * @param Request $request
     * @param InvoiceRepositoryContract $invoice
     * @return bool
     * @throws \Exception
     */
    public function charge(Request $request, InvoiceRepositoryContract $invoice): array
    {
        return [
            'provider' => 'bank-transfer',
            'amount' => $invoice->getOutstandingAmount(),
            'reference' => config('bank_transfer.reference_prefix') . $invoice->getId(),
            'details' => [],
            'confirmation_status' => config('bank_transfer.confirmation_status'),
            'confirmed_at' => config('bank_transfer.manually_confirm_payment_amount') ? null : now(),
        ];

    }

}
