<?php

namespace Mtc\BarclaycardEPDQ;

use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Checkout\Contracts\PayableContract;
use Mtc\Checkout\Contracts\PaymentGateway;
use Mtc\Checkout\PaymentForm;

/**
 * Barclaycard Smart Pay Payment Gateway
 *
 * @package  Mtc\BarclaycardEPDQ
 */
class BarclaycardEPDQ implements PaymentGateway {

    /**
     * Check if the gateway is available for use on this payment.
     *
     * @param InvoiceRepositoryContract $invoice
     * @param PayableContract $payable
     * @return bool
     */
    public function isApplicable(InvoiceRepositoryContract $invoice, $payable): bool
    {
        if ($invoice->getOutstandingAmount() <= 0.01) {
            return false;
        }

        return App::make(config('barclaycard-epdq.applicable_check_class'))->handle($invoice, $payable);
    }

    /**
     * Charge payment on invoice
     *
     * @param Request $request
     * @param InvoiceRepositoryContract $invoice
     * @return bool
     * @throws Exception
     */
    public function charge(Request $request, InvoiceRepositoryContract $invoice): array
    {

        return [
            'provider' => 'barclaycard-epdq',
            'currency_code' => $invoice->getCurrency(),
            'amount_in_currency' => $invoice->getOutstandingAmountInCurrency(),
            'confirmed_at' => now(),
        ];
    }
    
    /**
     * Render the payment template.
     *
     * @param InvoiceRepositoryContract $invoice
     * @return PaymentForm
     */
    public function getPaymentForm(InvoiceRepositoryContract $invoice): PaymentForm
    {

        $billing = $shipping = $invoice->getModel()->recipient;

        //- integration user details - //
        $PW = config('barclaycard-epdq.sha_pass_in');
        $PSPID = config('barclaycard-epdq.pspid');
        
        $barclay_card_info = [
            'AMOUNT' => ($invoice->getOutstandingAmount()*100),
            'CN' => trim($billing->first_name . ' ' . $billing->last_name),
            'COM' => trim(config('app.name') . ' Merchandise'),
            'CURRENCY' => trim($invoice->getCurrency()),
            'EMAIL' => trim($invoice->getEmail()),
            'LANGUAGE' => 'en_US',
            'ORDERID' => trim($invoice->getId()),
            'PSPID' => trim($PSPID),        
            /*'HOMEURL' => SITE_URL,
            'CATALOGURL ' => SITE_URL . '/browse',*/
            
            'ECOM_BILLTO_POSTAL_NAME_FIRST' => trim($billing->first_name),
            'ECOM_BILLTO_POSTAL_NAME_LAST' => trim($billing->last_name),
            
            'OWNERADDRESS'=>trim($shipping->address1 . ', ' . $shipping->address2),
            'OWNERCTY'=>trim($shipping->country ?? ''),
            'OWNERTELNO'=>trim($invoice->getModel()->contact_number),
            'OWNERTOWN'=>trim($shipping->city ?? ''),
            'OWNERZIP'=>trim($shipping->postcode ?? ''),
            
            /*
             * Following is the URL stuff
             */
            'ACCEPTURL' => route('charge_payment', [$invoice->getId(), 'gateway' => 'barclaycard-epdq']),
            'DECLINEURL' => route('failed_payment'),
            'EXCEPTIONURL' => route('failed_payment'),
            'CANCELURL' => route('failed_payment'),
            'BACKURL' => route('checkout.index'),
            
            /*
             * Style Stuff
             */
            'TITLE' => config('barclaycard-epdq.style.title'),
            'BGCOLOR' => config('barclaycard-epdq.style.bg_colour'),
            'TXTCOLOR' => config('barclaycard-epdq.style.txt_colour'),
            'TBLBGCOLOR' => config('barclaycard-epdq.style.table_bg_colour'),
            'TBLTXTCOLOR' => config('barclaycard-epdq.style.table_txt_colour'),
            'BUTTONBGCOLOR' => config('barclaycard-epdq.style.button_bg_colour'),
            'BUTTONTXTCOLOR' => config('barclaycard-epdq.style.button_txt_colour'),
            'FONTTYPE' => config('barclaycard-epdq.style.font_type'),
            'LOGO' => config('barclaycard-epdq.style.logo'),
        ];
        
        $string = '';
        
        //Keys need to be in alphabetical order
        ksort($barclay_card_info);

        $barclay_card_info = array_filter($barclay_card_info, fn($value) => !is_null($value) && $value !== '');
        
        foreach($barclay_card_info as $key => $value){
            $string .= strtoupper($key) . '=' . trim($value) .  $PW;
        }
        
        $type = '';
        
        switch (config('barclaycard-epdq.sha_encryption_type')) {
            case 'sha-1':
                $crypt = strtoupper(hash('sha1', $string));
                $type = 'sha-1';
                break;
            case 'sha-256':
                $crypt = strtoupper(hash('sha256', $string));
                $type = 'sha-256';
                break;
            case 'sha-512':
            default:
                $crypt = strtoupper(hash('sha512', $string));
                $type = 'sha-512';
                break;
        }
        
        /*var_dump($crypt);
        echo "</pre>";*/

        $barclay_card_info['SHASIGN'] = $crypt;

        $template = template('barclaycard-epdq/payment_form.twig', [
            'action' => 'https://mdepayments.epdq.co.uk/ncol/Test/orderstandard.asp',
            'form_fields' => [
                'info' => $barclay_card_info, 
                'debug' => [
                    'type_of_encryption' => $type,
                    'formatted_string' => $string,
                    'hashed' => $crypt,
                ]
            ],
        ]);

        return new PaymentForm($template, 'template', [
            'name' => __('barclaycard-epdq::barclaycard-epdq.payment_option_name'),
        ]);
    }
}