<?php

namespace Mtc\BasketRecovery\Models;


use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;
use Mtc\Basket\Basket;

/**
 * Class BasketRecoveryEmail
 *
 * @property $id
 * @property integer $basket_id
 * @property integer $basket_recovery_id
 * @property string $token
 * @property \Carbon\Carbon $created_at
 * @property \Carbon\Carbon $updated_at
 * @property-read \Mtc\Basket\Basket $basket
 * @property-read \Mtc\BasketRecovery\Models\BasketRecovery $basketRecovery
 *
 * @package Mtc\BasketRecovery\Models
 */
class BasketRecoveryEmail extends Model
{

    /**
     * @var string
     */
    protected $table = 'basket_recovery_sent_emails';

    /**
     * @var array
     */
    protected $fillable = [
        'basket_id',
        'recovery_id',
        'token'
    ];

    /**
     * @param \Mtc\Basket\Basket $basket
     * @param \Mtc\BasketRecovery\Models\BasketRecovery $recovery
     *
     * @return \Mtc\BasketRecovery\Models\BasketRecoveryEmail
     */
    public static function create(Basket $basket, BasketRecovery $recovery)
    {
        $email = new self([
            'token' => self::generateToken()
        ]);
        $email->basket()->associate($basket);
        $email->basketRecovery()->associate($recovery);
        $email->save();
        return $email;
    }

    /**
     * @return string
     */
    public static function generateToken()
    {
        return sha1(Str::random(32) . '-' . now()->timestamp);
    }

    /**
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function basket()
    {
        return $this->belongsTo(config('basket.model', Basket::class), 'basket_id');
    }

    /**
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function basketRecovery()
    {
        return $this->belongsTo(BasketRecovery::class, 'basket_recovery_id');
    }

    /**
     * @param $token
     *
     * @return \Mtc\BasketRecovery\Models\BasketRecoveryEmail|\Illuminate\Database\Eloquent\Model|null
     */
    public static function findByToken($token)
    {
        return self::query()->where('token', $token)->first();
    }

    /**
     * @param $token
     *
     * @return \Mtc\BasketRecovery\Models\BasketRecoveryEmail|\Illuminate\Database\Eloquent\Model
     */
    public static function findOrFailByToken($token)
    {
        return self::query()->where('token', $token)->firstOrFail();
    }

    /**
     * @return $this
     */
    public function markAsRead()
    {
        BasketRecoveryLog::query()
            ->firstOrCreate([
                'action' => 'read',
                'basket_id' => $this->basket_id,
                'basket_recovery_id' => $this->basket_recovery_id
            ]);

        return $this;
    }

    /**
     * @return $this
     */
    public function recordClick()
    {
        $this->log('click');
        return $this;
    }

    /**
     * @param $action
     * @param null $value
     *
     * @return $this
     */
    public function log($action, $value = null)
    {
        BasketRecoveryLog::create($this->basket, $this->basketRecovery, $action, $value);
        return $this;

    }

    /**
     * @return $this
     */
    public function restoreBasket()
    {
        session()->put('basket_id', $this->basket_id);
        return $this;
    }
}
