<?php

namespace Mtc\BasketRecovery\Http\Controllers\Admin;

use Illuminate\Database\Eloquent\ModelNotFoundException;
use Mtc\BasketRecovery\Http\Requests\Admin\PersistBasketRecoveryRequest;
use Mtc\BasketRecovery\Models\BasketRecovery;
use Mtc\Core\Http\Controllers\Controller;

/**
 * Class BasketRecoveryController
 *
 * @package Mtc\BasketRecovery\Http\Controllers\Admin
 * @property array $page_meta
 */
class BasketRecoveryController extends Controller
{

    /**
     * @return \Illuminate\View\Factory|\Illuminate\View\View
     */
    public function index()
    {

        $this->page_meta['title'] = 'Manage Basket Recovery Emails';

        $basket_recoveries = BasketRecovery::query()
            ->orderBy('is_active', 'desc')
            ->orderBy('interval_seconds', 'ASC')
            ->latest('updated_at')
            ->get();

        return template('BasketRecovery/admin/index.twig', [
            'basket_recoveries' => $basket_recoveries,
            'page_meta' => $this->page_meta
        ]);

    }

    /**
     * @return string
     */
    public function create()
    {

        $this->page_meta['title'] = 'Create Basket Recovery Email';

        return template('BasketRecovery/admin/edit.twig', [
            'basket_recovery' => $basket_recovery = new BasketRecovery,
            'action' => route('admin.basket-recovery.store'),
            'statistics' => $basket_recovery->getStatistics(),
            'page_meta' => $this->page_meta,
            'coupons' => config('basket_recovery.coupons.enabled', false)
        ]);

    }

    /**
     * @param \Mtc\BasketRecovery\Http\Requests\Admin\PersistBasketRecoveryRequest $request
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(PersistBasketRecoveryRequest $request)
    {

        $basket_recovery = new BasketRecovery($request->validated());
        $basket_recovery->setInterval($request->get('unit'), $request->get('measurement'));
        $basket_recovery->save();

        session()->flash('success', 'Successfully created the basket recovery email.');

        return redirect()->route('admin.basket-recovery.index');

    }

    /**
     * Edit resource
     *
     * @param integer $basket_recovery
     * @return string
     * @throws ModelNotFoundException
     */
    public function edit($basket_recovery)
    {

        $this->page_meta['title'] = 'Edit Basket Recovery Email';

        return template('BasketRecovery/admin/edit.twig', [
            'basket_recovery' => $basket_recovery = BasketRecovery::query()->findOrFail($basket_recovery),
            'action' => route('admin.basket-recovery.update', compact('basket_recovery')),
            'statistics' => $basket_recovery->getStatistics(),
            'page_meta' => $this->page_meta,
            'coupons' => config('basket_recovery.coupons.enabled', false)
        ]);

    }

    /**
     * @param \Mtc\BasketRecovery\Http\Requests\Admin\PersistBasketRecoveryRequest $request
     * @param integer $basket_recovery
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(PersistBasketRecoveryRequest $request, $basket_recovery)
    {

        $basket_recovery = BasketRecovery::query()->findOrFail($basket_recovery);

        $basket_recovery = $basket_recovery->fill($request->validated());
        $basket_recovery->setInterval($request->get('unit'), $request->get('measurement'));
        $basket_recovery->save();

        session()->flash('success', 'Successfully updated the basket recovery email.');

        return redirect()->route('admin.basket-recovery.index');

    }

    /**
     * @param $basket_recovery
     *
     * @return string
     * @throws \Exception
     */
    public function destroy($basket_recovery)
    {

        BasketRecovery::query()->findOrFail($basket_recovery)->delete();

        session()->flash('success', 'Successfully deleted the basket recovery email.');

        return redirect()->route('admin.basket-recovery.index');

    }

}
