<?php

namespace Mtc\Basket;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Event;
use Mtc\Basket\Events\LoadBasketExtensions;
use Mtc\Modules\Members\Classes\Member;
use Mtc\Plugins\NewsletterSignup\Classes\MailList;

/**
 * Class Basket
 *
 * @package Mtc\Basket
 */
class Basket extends Model
{
    use BasketValueCalculations;

    /**
     * We don't allow mass assigning basket id
     *
     * @var array
     */
    protected $guarded = [
        'id',
    ];

    /**
     * Cast attributes to specific types
     *
     * @var array
     */
    protected $casts = [
        'additional_info' => 'array',
        'tracking_data' => 'array',
    ];

    /**
     * @var array
     */
    protected $appends = [
        'cost_subtotal',
        'cost_total',
        'tax',
        'needs_delivery',
        'external_gateways',
        'backorder_cost_total',
    ];

    /**
     * Hide attributes from ajax response
     *
     * @var array
     */
    protected $hidden = [
        'tracking_data',
        'created_at',
        'updated_at',
    ];

    /**
     * Attach External gateways that w
     * @var array
     */
    public $external_gateways = [];

    /**
     * Extend model booting
     */
    protected static function boot()
    {
        parent::boot();

        static::creating(function (self $basket) {
            if (array_key_exists('additional_info', $basket->getDirty()) == false) {
                $basket->additional_info = [];
            }
        });


        static::created(function (self $basket) {
            // store the basket ID in session
            session()->put('basket_id', $basket->id);
        });

    }

    /**
     * Get the user this belongs to.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function member()
    {
        return $this->belongsTo(Member::class);
    }

    /**
     * Get the items associated with this basket.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function items()
    {
        return $this->hasMany(Item::class);
    }

    /**
     * Get the discounts associated with this basket.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function discounts()
    {
        return $this->hasMany(Discount::class);
    }

    /**
     * Get the surcharges associated with this basket.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function surcharges()
    {
        return $this->hasMany(Surcharge::class);
    }

    /**
     * Get the relationship to delivery information
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function deliverySurcharge()
    {
        $this->hasOne(Surcharge::class)
            ->where('surcharge_type', config('basket.delivery_surcharge_type'));
    }

    /**
     * Get the addresses associated with this basket.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function addresses()
    {
        return $this->hasMany(Address::class);
    }

    /**
     * Get the billing addresses associated with this basket.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function billingAddress()
    {
        return $this->hasOne(Address::class)
            ->where('type', Address::TYPE_BILLING);
    }
    /**
     * Get the shipping addresses associated with this basket.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function shippingAddress()
    {
        return $this->hasOne(Address::class)
            ->where('type', Address::TYPE_SHIPPING);
    }

    /**
     * Get the addresses associated with this basket.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function order()
    {
        return $this->hasOne(config('orders.order_class'))
            ->whereNotNull('paid_at');
    }
    /**
     * Get the addresses associated with this basket.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function unpaidOrders()
    {
        return $this->hasMany(config('orders.order_class'))
            ->whereNull('paid_at');
    }

    /**
     * Relationship with newsletter sign-up entries
     * This is to check if user has signed up for newsletters
     * to prevent from spamming unwarranted promotional emails
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function newsletterSignUp()
    {
        return $this->hasMany(MailList::class, 'email', 'email');
    }

    /**
     * Check if basket requires delivery
     *
     * @return bool
     */
    public function getNeedsDeliveryAttribute(): bool
    {
        if (config('basket.require_delivery_on_zero_weight_items')) {
            return true;
        }

        return $this->items
            ->filter(function (Item $item) {
                return $item->purchasable->requiresShipping();
            })
            ->isNotEmpty();
    }

    /**
     * Get the item count in basket
     * @return int
     */
    public function getItemCount()
    {
        return $this->items()->sum('quantity');
    }

    /**
     * Load discount / surcharge providers for basket
     *
     * @return \Illuminate\Support\Collection
     */
    public function loadBasketExtensions()
    {
        return collect(Event::dispatch(new LoadBasketExtensions($this)))
            ->filter();
    }

    /**
     * Get the External payment gateways
     *
     * @return array
     */
    public function getExternalGatewaysAttribute()
    {
        return $this->external_gateways ?? [];
    }
}
