<?php

namespace Mtc\Basket;

use Illuminate\Support\Facades\Config;
use Mtc\Foundation\Country;

trait BasketValueCalculations
{

    /**
     * Get the subtotal cost for all items in the basket.
     *
     * @return float
     */
    public function getSubtotalAttribute()
    {
        $item_tax = $this->doesVatApplyToBasket() && config('basket.discount_vat_on_basket') !== true
            ? $this->items->sum('line_tax')
            : 0;
        return $this->items->sum('line_total_ex_vat') + $item_tax;
    }

    /**
     * Get the subtotal cost for all items in the basket.
     *
     * @return float
     */
    public function getSubtotalExVatAttribute()
    {
        return $this->items->sum('line_total_ex_vat');
    }

    /**
     * Get the subtotal cost for all items in the basket.
     *
     * @return float
     */
    public function getCostSubtotalAttribute()
    {
        return config('basket.subtotal_display_with_vat') ? round($this->subtotal, 2) : round($this->subtotal_ex_vat, 2);
    }

    /**
     * Get the tax value for basket.
     *
     * @return float
     */
    public function getTaxAttribute()
    {
        $tax = 0;
        if ($this->doesVatApplyToBasket() === false) {
            return $tax;
        }

        // Ability to remove vat from basket completely
        if (config('basket.discount_vat_on_basket') !== true) {
            $tax += $this->items->sum('line_tax');
        }
        // Ability to remove vat from basket completely
        if (config('basket.discount_vat_on_additions') !== true) {
            $tax += $this->surcharges->sum('tax');
        }

        if (!config('basket.discount_display_with_vat', false)) {
            $tax -= $this->discounts->sum('tax');
        }

        return $tax;
    }

    /**
     * Calculate the total, including any price modifiers.
     *
     * @return float
     */
    public function getCostTotalAttribute()
    {
        $total = $this->subtotal_ex_vat
            + $this->tax
            + $this->surcharges->sum('price_ex_vat')
            - round($this->discounts->sum(config('basket.discount_display_with_vat', false) ? 'price' : 'price_ex_vat'), 2);

        return $total > 0 ? round($total, 2) : 0;
    }

    /**
     * Calculate the total ex vat, including any price modifiers.
     *
     * @return float
     */
    public function getCostTotalExVatAttribute()
    {
        $total = $this->subtotal_ex_vat
            + $this->surcharges->sum('price_ex_vat')
            - round($this->discounts->sum(config('basket.discount_display_with_vat', false) ? 'price' : 'price_ex_vat'), 2);

        return $total > 0 ? round($total, 2) : 0;
    }

    /**
     * Find the cost total for backorder items
     *
     * @return mixed
     */
    public function getBackorderCostTotalAttribute()
    {
        if (config('basket.allow_backorders') !== true) {
            return 0;
        }

        $backorder_subtotal = $this->items->sum('backorder_line_total_ex_vat');
        if ($backorder_subtotal == 0) {
            return 0;
        } elseif ($backorder_subtotal === $this->items->sum('line_total_ex_vat')) {
            // All items on backorder, surcharges and discounts are also on backorder
            return $this->getCostTotalAttribute();
        }

        $backorder_value_percentage = $backorder_subtotal / $this->subtotal_ex_vat;

        if (config('basket.surcharges_on_first_order_in_backorders')) {
            return $backorder_subtotal + $this->items->sum('backorder_line_tax');
        }

        return $backorder_subtotal
            + $this->tax * $backorder_value_percentage
            + $this->surcharges->sum('price_ex_vat') * $backorder_value_percentage
            - $this->discounts->sum(config('basket.discount_display_with_vat', false) ? 'price' : 'price_ex_vat') * $backorder_value_percentage;
    }

    /**
     * Check if basket is vat applicable
     * Determined by users shipping address
     *
     * return bool
     */
    protected function doesVatApplyToBasket()
    {

        if (config('basket.discount_vat_outside_default_region') !== true) {
            return true;
        }

        $destination = $this->shippingAddress;

        // Address yet not set - not viable for discounted vat
        if (empty($destination->country)) {
            return true;
        }

        $destination_within_applicable_region = Country::query()
            ->where(config('basket.vat_applicable_region_conditions', []))
            ->where('code', $destination->country)
            ->exists();

        if ($destination_within_applicable_region) {
            return true;
        }

        if (config('basket.vat_does_not_apply_to_channel_islands')) {
            // Only allow JE & GY postcodes for UK based no_vat zones
            $postcode = strtoupper(substr($destination->postcode, 0, 2));
            if ($destination->country === 'GB' && !in_array($postcode, config('foundation.channel_island_postcodes'))) {
                return true;
            }
        }

        Config::set('basket.subtotal_display_with_vat', false);

        return false;
    }
}
