<?php

namespace Mtc\Basket\Http\Resources;

use Illuminate\Http\Resources\Json\JsonResource;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Event;
use Mtc\Basket\Basket;
use Mtc\Basket\Contracts\BasketRepositoryInterface;
use Mtc\ShippingManager\Facades\DeliveryMethod;
use Mtc\ShippingManager\Facades\Shipping;

/**
 * Class BasketAjax
 *
 * @package Mtc\Basket\Http\Resources
 */
class BasketAjax extends JsonResource
{
    /**
     * BasketAjax constructor.
     *
     * The resource is current basket (loaded from facade)
     */
    public function __construct()
    {
        parent::__construct(App::make(BasketRepositoryInterface::class));
    }

    /**
     * Transform the resource into an array.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return array
     */
    public function toArray($request)
    {
        if (config('shipping_manager.preemptive_shipping_calculation')) {
            Shipping::estimateShippingCost($this->resource);
        }

        Event::dispatch(BasketAjax::class, $this->resource);
        /** @var Basket resource */
        $this->resource = $this->resource->getModel();
        $this->resource->load([
            'items.purchasable',
            'items.lines',
            'surcharges',
            'discounts',
        ]);

        // Make sure purcahsable exists on site
        $this->resource->items
            ->filter(function ($item) {
                return $item->purchasable !== null;
            })
            ->each(function ($item) {
                $item->setHidden([ 'purchasable' ]);
            });

        $this->resource->validation = $this->validateBasket();

        return parent::toArray($request);
    }

    /**
     * Check which sections are valid for basket
     */
    public function validateBasket()
    {
        return collect(config('basket.basket_validation', []))
            ->map(function ($validator_class) {
                return App::make($validator_class)->handle();
            });
    }
}
