<?php

namespace Mtc\Basket\Contracts;

use Illuminate\Database\Eloquent\Builder;

/**
 * This interface defines the necessary methods for building a Basket
 *
 * @package  Mtc\Basket
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
interface Purchasable
{
    /**
     * Whether this item requires shipping
     *
     * @return mixed
     */
    public function requiresShipping();

    /**
     * Check whether purchasable is available
     *
     * @return bool
     */
    public function isAvailable(): bool;

    /**
     * Get the weight of the purchasable item
     *
     * @return mixed
     */
    public function getWeight();

    /**
     * Url to view purchasable item
     *
     * @return mixed
     */
    public function getUrl();

    /**
     * Price for which the item can be purchased
     *
     * @return mixed
     */
    public function getBasketImage();

    /**
     * Price for which the item can be purchased
     *
     * @return mixed
     */
    public function getOriginalPrice();

    /**
     * Check if price has a discount
     *
     * @return mixed
     */
    public function hasDiscountedPrice();

    /**
     * Price for which the item can be purchased
     *
     * @return mixed
     */
    public function getPrice();

    /**
     * Name for the basket
     *
     * @return mixed
     */
    public function getName();

    /**
     * Get the Sku code of attribute
     *
     * @return mixed
     */
    public function getSku();

    /**
     * Get the available stock level
     *
     * @return int
     */
    public function getStock();

    /**
     * Max quantity allowed of this item in basket
     *
     * @return mixed
     */
    public function getMaxQuantity();

    /**
     * Fields that identify attributes that should be displayed in basket alongside name
     *
     * @return mixed
     */
    public function getAttributeFields(): array;

    /**
     * Fields that identify attributes that should be displayed in basket alongside name
     *
     * @return mixed
     */
    public function only($attributes);

    /**
     * Purchasable should have the ability to be found by term
     *
     * @param Builder $query
     * @param string $term
     * @return mixed
     */
    public function scopeSearchByTerm(Builder $query, string $term);

    /**
     * Purchasable must have a search name
     *
     * @return string
     */
    public function getSearchNameAttribute(): string;
}
