<?php

namespace Mtc\Basket\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Mtc\Basket\Contracts\BasketRepositoryInterface;
use Mtc\Basket\Contracts\Purchasable;
use Mtc\Basket\Http\Requests\AddBasketItem;
use Mtc\Basket\Http\Requests\UpdateBasketItem;
use Mtc\Core\Node;
use Mtc\ShippingManager\Facades\DeliveryMethod;

/**
 * Class BasketController
 *
 * @package Mtc\Basket\Http\Controllers
 */
class BasketController extends Controller
{
    /**
     * Show basket template
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function index(Request $request, BasketRepositoryInterface $basket)
    {
        return App::make(config('basket.template_response_class'))->basket($request, $basket);
    }

    /**
     * Return ajax response with basket contents
     *
     * @return mixed
     */
    public function show()
    {
        App::make(BasketRepositoryInterface::class)->allowOnlyAvailableItems();
        return App::make(config('basket.ajax_response_class'));
    }

    /**
     * Add item to basket
     *
     * @param AddBasketItem $request
     */
    public function store(AddBasketItem $request, BasketRepositoryInterface $basket)
    {
        /** @var Purchasable $node */
        $purchasable = Node::query()->findOrFail($request->input('node_id'))->nodeable;
        $basket->addItem($purchasable, $request->input('quantity'), config('basket.append_quantity', true));

        return [
            'status' => 'ok',
            'message' => __('basket.item_added', ['url' => route('basket.index')]),
            'basket' => App::make(config('basket.ajax_response_class')),
        ];
    }

    /**
     * Update basket stock entries
     *
     * @param UpdateBasketItem $request
     */
    public function update(UpdateBasketItem $request, BasketRepositoryInterface $basket)
    {
        $basket->updateItemQuantities($request->input('stock_amends'));
        return App::make(config('basket.ajax_response_class'));
    }

    /**
     * Update basket stock entries
     *
     * @param UpdateBasketItem $request
     */
    public function updatePrice(Request $request, BasketRepositoryInterface $basket)
    {
        $basket->setCustomPrice($request->input('id'), $request->input('price'));
        return App::make(config('basket.ajax_response_class'));
    }

    /**
     * Remove Discount from basket discounts
     *
     * @param Request $request
     * @param BasketRepositoryInterface $basket
     * @return mixed
     */
    public function destroy(Request $request, BasketRepositoryInterface $basket)
    {
        $basket->removeDiscount($request->input('discount_id'));

        if (config('shipping_manager.preemptive_shipping_calculation')) {
            DeliveryMethod::methodsForBasket($basket);
        }
        return App::make(config('basket.ajax_response_class'));
    }
}
