<?php

namespace Mtc\Basket\Http\Resources;

use Illuminate\Http\Resources\Json\JsonResource;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Event;
use Mtc\Basket\Contracts\BasketRepositoryInterface;
use Mtc\ShippingManager\Facades\Shipping;

/**
 * Class CheckoutAjax
 *
 * @package Mtc\Basket\Http\Resources
 */
class CheckoutAjax extends JsonResource
{
    /**
     *
     * @var BasketRepositoryInterface
     */
    protected $repository;

    /**
     * BasketAjax constructor.
     *
     * The resource is current basket (loaded from facade)
     */
    public function __construct(BasketRepositoryInterface $basket_repository)
    {
        parent::__construct($basket_repository->getModel());
        $this->repository = $basket_repository;
    }

    /**
     * Transform the resource into an array.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return array
     */
    public function toArray($request)
    {
        Event::dispatch(CheckoutAjax::class, $this->resource);
        $this->resource->load([
            'items.purchasable',
            'items.lines',
            'surcharges',
            'discounts',
            'addresses',
        ]);

        // Make sure purcahsable exists on site
        $this->resource->items
            ->filter(function ($item) {
                return $item->purchasable !== null;
            });

        $this->resource->validation = $this->validateBasket();
        $this->resource->sections = $this->allowedSections();
        $this->resource->errors = $this->repository->getErrors();

        return parent::toArray($request);
    }

    /**
     * Check which sections are valid for basket
     */
    public function validateBasket()
    {
        return collect(config('basket.checkout_validation', []))
            ->map(function ($validator_class) {
                return App::make($validator_class)->handle();
            });
    }

    /**
     * Set up what sections are allowed in the checkout
     *
     * @return array
     */
    protected  function allowedSections()
    {
        $enabled_types = Shipping::enabledTypes();

        $shipping_types = $enabled_types
            ->map(function ($data, $type) {
                return App::make(config('basket.allow_checkout_sections.shipping_type'))->handle($type);
            })->toArray();


        return collect(config('basket.allow_checkout_sections', []))
            ->reject(function ($validator_class, $validator_type) {
                return $validator_type === 'shipping_type';
            })
            ->map(function ($validator_class) {
                return App::make($validator_class)->handle();
            })
            ->merge($shipping_types);

    }
}
