<?php

namespace Mtc\Basket;

use Illuminate\Database\Eloquent\Model;

/**
 * Class Discount
 *
 * @package Mtc\Basket
 */
class Discount extends Model
{
    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'basket_discounts';

    /**
     * Columns that cannot be mass assigned
     *
     * @var array
     */
    protected $guarded = [
        'id'
    ];

    /**
     * Touch (update) relationship when changes are made
     *
     * @var array
     */
    protected $touches = [
        'basket',
    ];

    /**
     * Hide attributes from ajax response
     *
     * @var array
     */
    protected $hidden = [
        'basket_id',
        'discount_id',
        'created_at',
        'updated_at',
        'discount',
        'applicable_items',
    ];

    /**
     * Append attributes to JSON response
     *
     * @var array
     */
    protected $appends = [
        'name',
        'price',
        'is_removable',
    ];

    /**
     * Cast attributes to specific types
     *
     * @var array
     */
    protected $casts = [
        'applicable_items' => 'array',
    ];

    /**
     * Relationship with Basket
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function basket()
    {
        return $this->belongsTo(Basket::class);
    }

    /**
     * Relationship with discount
     *
     * @return \Illuminate\Database\Eloquent\Relations\MorphTo
     */
    public function discount()
    {
        return $this->morphTo('discount');
    }

    /**
     * Show name attribute from surcharge
     *
     * @return mixed
     */
    public function getNameAttribute()
    {
        return $this->discount->getNameAttribute();
    }

    /**
     * Show discount value
     *
     * @return mixed
     */
    public function getPriceExVatAttribute()
    {
        return $this->discount->getAmountAttribute()->raw(false);
    }

    /**
     * Show discount value
     *
     * @return mixed
     */
    public function getPriceAttribute($with_vat = null)
    {
        $with_vat = $with_vat ?? config('basket.discount_display_with_vat');
        return $this->discount->getAmountAttribute()->raw($with_vat);
    }

    /**
     * Show price tax value
     *
     * We are using null coalescence here to deal with
     * null being passed as a value
     *
     * @param int $precision
     * @return mixed
     */
    public function getTaxAttribute($precision = 3)
    {
        return $this->discount->getAmountAttribute()->tax($precision ?? 3);
    }

    /**
     * Whether discount can be removed from basket
     *
     * @return mixed
     */
    public function getIsRemovableAttribute()
    {
        return $this->discount->getIsRemovableAttribute();
    }

    /**
     * Show vat rate of the discount
     *
     * @return mixed
     */
    public function vatRate()
    {
        if ($this->basket->doesVatApplyToBasket()) {
            return $this->surcharge->getPriceAttribute()->taxRate();
        }

        return array_search(0, config('tax.vat_rates')) ?? 0;
    }

}
