<?php

namespace Mtc\Basket\Providers;

use Illuminate\Support\Facades\Event;
use Mtc\Basket\Console\Commands\InstallBasket;
use Mtc\Basket\Contracts\BasketRepositoryInterface;
use Mtc\Basket\Contracts\SetAddressRequest;
use Mtc\Basket\Facades\Basket as BasketFacade;
use Mtc\Basket\Contracts\BasketDetailsRequestContract;
use Illuminate\Foundation\Support\Providers\EventServiceProvider;
use Mtc\Basket\PurchasableManager;
use Mtc\Basket\Events\AddExternalPaymentGatewayScripts;

/**
 * Class BasketServiceProvider
 * @package Mtc\Basket\Providers
 */
class BasketServiceProvider extends EventServiceProvider
{
    /**
     * Bootstrap the application services.
     */
    public function boot()
    {
        parent::boot();

        /*
         * Optional methods to load your package assets
         */
        $this->loadTranslationsFrom(dirname(__DIR__, 2) . '/resources/lang', 'basket');
        $this->loadMigrationsFrom(dirname(__DIR__, 2) . '/database/migrations');


        if ($this->app->runningInConsole()) {
            $this->publishes([
                dirname(__DIR__, 2) . '/config/basket.php' => config_path('basket.php'),
            ], 'config');

            // Publishing assets.
            $this->publishes([
                dirname(__DIR__, 2) . '/resources/less' => resource_path('less/vendor'),
            ], 'assets');
            // Publishing assets.
            $this->publishes([
                dirname(__DIR__, 2) . '/resources/js' => resource_path('js/vendor'),
            ], 'assets');

            // Publishing the translation files.
            $this->publishes([
                dirname(__DIR__, 2) . '/resources/lang' => resource_path('lang/vendor/basket'),
            ], 'lang');
        }

        $this->bootTwigDependencies();
    }

    /**
     * Register the application services.
     */
    public function register()
    {
        // Automatically apply the package configuration
        $this->mergeConfigFrom(dirname(__DIR__, 2) . '/config/basket.php', 'basket');

        collect(config('basket.service_providers', []))
            ->each(function ($service_provider) {
                $this->app->register($service_provider);
            });

        // Load a custom basket (or default).
        $this->app->singleton('basket', function ($app) {
            return $this->app->make(config('basket.basket_loader_class'))->__invoke();
        });

        // Load a custom basket (or default).
        $this->app->singleton(BasketRepositoryInterface::class, function ($app) {
            return $this->app->make(config('basket.repository_class'));
        });

        // Register the main class to use with the facade
        $this->app->bind('product_search_response', config('basket.product_search_response'));

        // Register the main class to use with the facade
        $this->app->bind('basketAjaxResponse', config('basket.ajax_response_class'));

        $this->app->bind(BasketDetailsRequestContract::class, config('basket.details_request_validation'));

        $this->app->bind(SetAddressRequest::class, config('basket.set_address_request'));

        // Register the main class to use with the facade
        $this->app->singleton('purchasable', function ($app) {
            return new PurchasableManager($app);
        });

    }

    /**
     * Boot twig functionality
     */
    public function bootTwigDependencies()
    {
        app('twig_loader')->addPath(dirname(__DIR__, 2) . '/resources/templates');

        app('twig')->addFunction(new \Twig_SimpleFunction('basketItemCount', function () {
            return BasketFacade::getItemCount();
        }));

        app('twig')->addFunction(new \Twig_SimpleFunction('basketExtensions', function () {
            return BasketFacade::loadBasketExtensions();
        }));

        if (!empty(config('checkout'))) {
            app('twig')->addFunction(new \Twig_SimpleFunction('addExternalPaymentGatewayScripts', function () {
                return Event::dispatch(new AddExternalPaymentGatewayScripts());
            }));
        }
    }

    /**
     * Events to register
     *
     * @return array|\Illuminate\Config\Repository|mixed
     */
    public function listens()
    {
        return config('basket.events');
    }
}
