<?php

namespace Mtc\Basket;

use Illuminate\Support\Facades\Validator;
use Mtc\Foundation\Rules\ValidPhoneNumber;
use Mtc\Foundation\Rules\ValidPostcode;
use Mtc\ShippingManager\Facades\Shipping;

/**
 * Class BasketValidator
 *
 * @package Mtc\Basket
 */
class CheckoutValidator extends BasketValidator
{
    /**
     * Validate the basket
     *
     * @return array
     */
    public function validate()
    {
        return [
            'details' => $this->checkDetails(),
            'items' => $this->checkItems(),
            'shipping_address' => $this->checkShippingAddress(),
            'delivery_method' => $this->checkDelivery(),
            'surcharges' => $this->checkSurcharges(),
            'discounts' => $this->checkDiscounts(),
            'billing_address' => $this->checkBillingAddress(),
        ];
    }

    /**
     * Check if there is delivery address error
     *
     * @return bool
     */
    protected function checkDetails()
    {
        $validator = Validator::make($this->basket->toArray(), [
            'email' => 'required|email',
            'contact_number' => new ValidPhoneNumber(),
        ]);

        return $validator->passes();
    }

    /**
     * Check if there is delivery address error
     *
     * @return bool
     */
    protected function checkShippingAddress()
    {
        if (!$this->basket->needs_delivery) {
            return true;
        }

        if ($this->basket->shipping_type === 'delivery') {
            $address = $this->basket->shippingAddress ? $this->basket->shippingAddress->toArray() : [];
            $validator = Validator::make($address, [
                'type' => 'required',
                'first_name' => 'required',
                'last_name' => 'required',
                'address1' => 'required',
                'city' => 'required',
                'country' => 'required|exists:countries,code',
                'postcode' => new ValidPostcode($address['country'] ?? ''),
            ]);

            return $validator->passes();
        }

        return $this->basket->surcharges->where('surcharge_type', 'collection')->isNotEmpty();
    }

    /**
     * Check if delivery method is set up
     *
     * @return bool
     */
    protected function checkDelivery()
    {
        if (!$this->basket->needs_delivery) {
            return true;
        }

        $keys = Shipping::enabledTypes()->keys();
        return $keys->filter(function ($key) {
            return $this->repository->hasSurcharge(0, $key);
        })
            ->isNotEmpty();
    }

    /**
     * Check if basket has valid billing address
     *
     * @return mixed
     */
    protected function checkBillingAddress()
    {
        $address = $this->basket->billingAddress ? $this->basket->billingAddress->toArray() : [];
        $validator = Validator::make($address, [
            'type' => 'required',
            'first_name' => 'required',
            'last_name' => 'required',
            'address1' => 'required',
            'city' => 'required',
            'country' => 'required|exists:countries,code',
            'postcode' => new ValidPostcode($address['country'] ?? ''),
        ]);

        return $validator->passes();
    }

    /**
     * Check if Surcharges are valid
     *
     * @return bool
     */
    protected function checkSurcharges()
    {
        return true;
    }

    /**
     * Check if discounts are valid
     *
     * @return bool
     */
    protected function checkDiscounts()
    {
        return true;
    }

    /**
     * Check if stock values are ok for the basket
     *
     * @return bool
     */
    protected function checkItemsOutOfStock()
    {
        return collect($this->basket->items)
            ->filter(function ($item) {
                return $item->purchasable->getStock() < $item->quantity;
            })
            ->count() == 0;
    }
}
