<?php

namespace Mtc\Basket\Contracts;

use Illuminate\Database\Eloquent\Model;

/**
 * Interface BasketRepositoryInterface
 *
 * @package Mtc\Basket
 */
interface BasketRepositoryInterface
{
    /**
     * Set address info on Basket
     *
     * @param $type
     * @param $address_data
     */
    public function setAddress($type, $address_data);

    /**
     * Set generic info on Basket (e.g. email/phone)
     *
     * @param $details
     */
    public function setDetails($details);

    /**
     * Check if basket has items
     *
     * @return bool
     */
    public function hasItems(): bool;

    /**
     * Fetch the id
     *
     * @return int
     */
    public function getId(): int;

    /**
     * Fetch the member associated with basket
     *
     * @return int
     */
    public function getMember();

    /**
     * Find a basket by ID
     *
     * @param $basket_id
     */
    public function find($basket_id);

    /**
     * Fetch the current basket model
     *
     * @return Model
     */
    public function getModel(): Model;

    /**
     * Add an item to the basket. If basket doesn't exist, it will create it.
     *
     * @param Purchasable $purchasable item that implements Purchasable contract
     * @param integer $quantity Number of items to add to basket
     * @param bool $append Flag whether the quantities of the basket items should be appended
     * @param bool $force_new Flag to create the basket item as a new line
     * @return void
     */
    public function addItem(Purchasable $purchasable, int $quantity, $append = true, $force_new = false);

    /**
     * Check if this item is in basket
     *
     * @param Purchasable $purchasable item that implements Purchasable contract
     * @param integer $quantity Number of items to add to basket
     * @return bool
     */
    public function hasItem(Purchasable $purchasable, $quantity = false);

    /**
     * Get the items in basket
     *
     * @return mixed
     */
    public function getItems();

    /**
     * Change the quantities of basket items based on id => stock array
     *
     * @param array $quantity_array
     * @return mixed
     */
    public function updateItemQuantities($quantity_array);

    /**
     * Set custom basket price for item - e.g. fixed price in order creation
     *
     * @param int $item_id
     * @param float $item_price
     */
    public function setCustomPrice($item_id, $item_price);

    /**
     * Add discount to current Basket
     *
     * @param int $discount_id
     * @param string $discount_class
     * @param array|null $applicable_items
     * @return void
     */
    public function addDiscount(int $discount_id, string $discount_class, array $applicable_items = null);

    /**
     * Check if Basket has discount
     *
     * @param int $discount_id
     * @param string $discount_class
     * @return bool
     */
    public function hasDiscount(int $discount_id, string $discount_class): bool;

    /**
     * Remove discount from current basket
     *
     * @param int $discount_id
     * @return void
     */
    public function removeDiscount(int $discount_id);

    /**
     * Add surcharge to current basket
     *
     * @param int $surcharge_id
     * @param string $surcharge_class
     * @return void
     */
    public function addSurcharge(int $surcharge_id, string $surcharge_class);

    /**
     * Get the surcharge ID from type
     *
     * @param string $surcharge_type
     * @return int|null
     */
    public function getSurchargeIdFromType($surcharge_type);

    /**
     * Check if basket has surcharge
     *
     * @param int $surcharge_id
     * @param string $surcharge_class
     * @return bool
     */
    public function hasSurcharge(int $surcharge_id, string $surcharge_class): bool;

    /**
     * Update surcharge for surcharge type
     *
     * @param string $surcharge_type
     * @param int $surcharge_id
     * @return void
     */
    public function updateSurchargeByType($surcharge_type, $surcharge_id);

    /**
     * Remove surcharge from current basket
     *
     * @param string $surcharge_type
     * @return void
     */
    public function removeSurchargeByType($surcharge_type);

    /**
     * Remove surcharge from current basket
     *
     * @param int $surcharge_id
     * @return void
     */
    public function removeSurcharge(int $surcharge_id);

    /**
     * Check if basket needs delivery
     *
     * @return bool
     */
    public function needsDelivery();

    /**
     * Get the subtotal cost for all items in the basket.
     *
     * @return float
     */
    public function getCostSubtotalAttribute();

    /**
     * Get the total cost for all items in the basket.
     *
     * @return float
     */
    public function getCostTotalAttribute();

    /**
     * Get the basket weight.
     *
     * @return float
     */
    public function getWeight();

    /**
     * Get the destination address of the basket
     *
     * @return float
     */
    public function getDestinationAddress();

    /**
     * Get the customer name for this basket
     *
     * @return string
     */
    public function getCustomerName(): string;

    /**
     * Get the vat rate applied to basket
     *
     * @return string
     */
    public function getVatValue(): string;

    /**
     * Get the basket type
     *
     * @return mixed
     */
    public function getType();

    /**
     * Set the member ID on basket
     *
     * @param $member_id
     * @return void
     */
    public function setMember($member_id);

    /**
     * Add external gateway to basket
     *
     * @param $name
     * @param $context
     * @return void
     */
    public function setExternalGateway($name, $context);

    /**
     * Get the baskets currency
     *
     * @return string
     */
    public function getCurrency(): string;

    /**
     * Basket needs to ensure that only existing items are allowed in basket
     *
     * @return mixed
     */
    public function allowOnlyAvailableItems();

    /**
     * Whether Ex vat costs should be used.
     * Necessary for order generation process
     *
     * @return bool
     */
    public function shouldUseExVat(): bool;
}
