<?php
namespace Mtc\BigCommerceRates\Livewire\BigCommerce;

use Livewire\Component;
use Mtc\BigCommerceRates\Models\BigCommerce\Rates\StoreCarrier;
use Mtc\BigCommerceRates\Models\BigCommerce\Rates\RateDetails;
use Mtc\BigCommerceRates\Models\BigCommerce\Rates\RateCost;
use Mtc\BigCommerceRates\Models\StoreInstallation;
use Illuminate\Support\Facades\Http;
use Livewire\WithPagination;
use Illuminate\Pagination\LengthAwarePaginator;


class Carrier extends Component
{
    use WithPagination;

    public $storeHash;
    public $expandedCarrier = null;
    public $showModal = false;
    public $selectedCarrier = null;
    public $selectedCarrierRatesPage = 1;
    public $carriersPage;
    public $ratesPage;
    public $currencies = [];
    // Stepper
    public $step = 1;

    // Carrier fields
    public $carrier_info_code = '';
    public $carrier_info_display_name = '';

    // Rates (for creation)
    public $rates = [];
    public $rate_code = '';
    public $rate_description = '';
    public $rate_currency = '';
    public $rate_amount = '';

    // Modal rate fields
    public $modal_rate_code = '';
    public $modal_rate_description = '';
    public $modal_rate_currency = '';
    public $modal_rate_amount = '';
    public $showAddRateForm = false;

    protected $rules = [
        'carrier_info_code' => 'required|string|max:255',
        'carrier_info_display_name' => 'required|string|max:255',
    ];

    public function mount($storeHash = null)
    {
        $this->storeHash = $storeHash
            ?? request()->query('storeHash')
            ?? session('storeHash')
            ?? (auth()->user()?->storeInstallation?->store_hash ?? null);
         $this->getCarriersProperty();


        $this->currencies = $this->fetchStoreCurrencies($this->storeHash);
    }

    protected function fetchStoreCurrencies($storeHash)
    {
        $endpointUrl = env('APP_URL')."bc-api/{$storeHash}/v2/currencies";
        try {
            $response = Http::timeout(400)
                ->connectTimeout(1)
                ->get($endpointUrl)
                ->collect();
            return $response->pluck('currency_code')->all();
        } catch (\Illuminate\Http\Client\HttpClientException $clientException) {
            session()->flash('error', "HTTP Client Exception: " . $clientException->getMessage());
            return 0;
        }
    }

    public function getCarriersProperty()
    {
        $store = StoreInstallation::findByStoreHash($this->storeHash);

        if ($store) {
            $query = StoreCarrier::where('store_id', $store->id)
                ->with(['rates', 'rates.rateCost']);

            // If there are no carriers, return an empty paginator
            if (!$query->exists()) {
                return new LengthAwarePaginator([], 0, 8, 1, [
                    'path' => request()->url(),
                    'pageName' => 'carriersPage',
                ]);
            }

            return $query->paginate(8, ['*'], 'carriersPage');
        }

        // Always return a paginator, even if empty
        return new LengthAwarePaginator([], 0, 8, 1, [
            'path' => request()->url(),
            'pageName' => 'carriersPage',
        ]);
    }

    public function showCarrierModal($carrierId)
    {
        $this->selectedCarrier = StoreCarrier::with(['rates', 'rates.rateCost'])->find($carrierId);
        $this->selectedCarrierRatesPage = 1;
        $this->showModal = true;
        $this->showAddRateForm = false;
    }

    public function closeModal()
    {
        $this->showModal = false;
        $this->selectedCarrier = null;
    }

    public function getPaginatedRatesProperty()
    {
        if (!$this->selectedCarrier) {
            return new LengthAwarePaginator([], 0, 8, 1, [
                'path' => request()->url(),
                'pageName' => 'ratesPage',
            ]);
        }
        $rates = $this->selectedCarrier->rates;
        $perPage = 8;
        $page = $this->selectedCarrierRatesPage;
        $offset = ($page - 1) * $perPage;
        $paginated = new LengthAwarePaginator(
            $rates->slice($offset, $perPage)->values(),
            $rates->count(),
            $perPage,
            $page,
            [
                'path' => request()->url(),
                'pageName' => 'ratesPage',
            ]
        );
        return $paginated;
    }

    public function updatingCarriersPage()
    {
        $this->resetPage('carriersPage');
    }

    public function updatingSelectedCarrierRatesPage()
    {
        $this->resetPage('ratesPage');
    }

    // Stepper methods
    public function nextStep()
    {
        if ($this->step === 1) {
            $this->validate();
        }
        $this->step++;
    }

    public function prevStep()
    {
        $this->step--;
    }

    // Rate management
    public function addRate()
    {
        $this->rates[] = [
            'code' => $this->rate_code,
            'description' => $this->rate_description,
            'currency' => $this->rate_currency,
            'amount' => $this->rate_amount,
        ];
        $this->rate_code = '';
        $this->rate_description = '';
        $this->rate_currency = '';
        $this->rate_amount = '';
    }

    public function removeRate($index)
    {
        unset($this->rates[$index]);
        $this->rates = array_values($this->rates);
    }

    public function saveCarrierWithRates()
    {
        $this->validate();
        $store = StoreInstallation::findByStoreHash($this->storeHash);
        if (!$store) {
            $this->addError('store', 'Store not found.');
            return;
        }
        // Create carrier
        $carrier = StoreCarrier::create([
            'store_id' => $store->id,
            'carrier_info_code' => $this->carrier_info_code,
            'carrier_info_display_name' => $this->carrier_info_display_name,
        ]);
        // Create rates
        foreach ($this->rates as $rate) {

            $rateDetail = RateDetails::create([
                'code' => $rate['code'],
                'display_name' => $rate['description'],
                'carrier_id' => $carrier->id,
                'rate_id' => uuid_create(UUID_TYPE_TIME),
            ]);
            RateCost::create([
                'rate_details_id' => $rateDetail->id,
                'currency' => $rate['currency'],
                'amount' => $rate['amount'],
            ]);
        }
        // Reset
        $this->carrier_info_code = '';
        $this->carrier_info_display_name = '';
        $this->rates = [];
        $this->step = 1;
        $this->getCarriersProperty();
    }

    public function toggleExpand($carrierId)
    {
        $this->expandedCarrier = $this->expandedCarrier === $carrierId ? null : $carrierId;
    }

    public function deleteCarrier($carrierId)
    {
        $carrier = StoreCarrier::find($carrierId);
        if ($carrier) {
            // Delete all related rates and costs
            if ($carrier->rateDetail) {
                if ($carrier->rateDetail->rateCost) {
                    $carrier->rateDetail->rateCost->delete();
                }
                $carrier->rateDetail->delete();
            }
            $carrier->delete();
        }
        $this->getCarriersProperty();
    }

    public function deleteRate($rateId)
    {
        $rate = RateDetails::find($rateId);
        if ($rate) {
            if ($rate->rateCost) {
                $rate->rateCost->delete();
            }
            $rate->delete();
        }
        $this->getCarriersProperty();
    }

    public function addRateToCarrier()
    {
        $this->validate([
            'modal_rate_code' => 'required|string|max:255',
            'modal_rate_description' => 'required|string|max:255',
            'modal_rate_currency' => 'required|string|max:10',
            'modal_rate_amount' => 'required|numeric|min:0',
        ]);
        if (!$this->selectedCarrier) return;
        $rateDetail = RateDetails::create([
            'code' => $this->modal_rate_code,
            'display_name' => $this->modal_rate_description,
            'carrier_id' => $this->selectedCarrier->id,
            'rate_id' => uuid_create(UUID_TYPE_TIME),
        ]);
        RateCost::create([
            'rate_details_id' => $rateDetail->id,
            'currency' => $this->modal_rate_currency,
            'amount' => $this->modal_rate_amount,
        ]);
        // Reset modal fields
        $this->modal_rate_code = '';
        $this->modal_rate_description = '';
        $this->modal_rate_currency = '';
        $this->modal_rate_amount = '';
        // Refresh selected carrier
        $this->selectedCarrier = StoreCarrier::with(['rates', 'rates.rateCost'])->find($this->selectedCarrier->id);
    }

    public function finishAddingRates()
    {
        // Optionally refresh the carrier list or close the modal
        $this->selectedCarrier = StoreCarrier::with(['rates', 'rates.rateCost'])->find($this->selectedCarrier->id);
        $this->showModal = false;
    }

    public function toggleAddRateForm()
    {
        $this->showAddRateForm = !$this->showAddRateForm;
    }

    public function render()
    {

        return view('bigcommerce::livewire.bigcommerce.rates.carrier', [
            'carriers' => $this->getCarriersProperty(),
            'paginatedRates' => $this->getPaginatedRatesProperty(),
        ]);
    }
}
