<?php

namespace Mtc\BigCommerceRates\Http\Controllers\BigCommerce\App;

use App\Http\Controllers\Controller;
use Mtc\BigCommerceRates\Configuration\Configuration;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Log;
use Mtc\BigCommerceRates\Models\StoreInstallation;
use Mtc\BigCommerceRates\Models\User;
use Tymon\JWTAuth\Facades\JWTAuth;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;

class Install extends Controller
{
    public function __construct(
        protected Configuration $configuration,
        protected Request $request
    ) {
        $this->baseURL = env("APP_URL");
    }

    public function uninstall()
    {
        Log::info("Uninstalling");
        Log::debug($this->request->all());

        $signedPayload = $this->request->input('signed_payload');
        if (!empty($signedPayload)) {
            $verifiedData = $this->verifySignedRequest($signedPayload);
            if ($verifiedData !== null) {
                $storeHash = str_replace('stores/', '', $verifiedData['context']);
                StoreInstallation::deleteByHash($storeHash);
            }
        }
    }

    public function install(): RedirectResponse
    {
        // Ensure necessary parameters are passed
        if (!$this->request->has(['code', 'scope', 'context'])) {
            return redirect()->route('error', ['error_message' => 'Not enough information was passed to install this app.']);
        }

        // Log the received parameters for debugging
        Log::info('Code: ' . $this->request->input('code'));
        Log::info('Scope: ' . $this->request->input('scope'));
        Log::info('Context: ' . $this->request->input('context'));

        try {

            $object = [
                'client_id' => $this->configuration->getAppClientId(),
                'client_secret' => $this->configuration->getAppSecret(),
                'redirect_uri' => $this->redirectHome() . 'biginstall/install',
                'code' => $this->request->input('code'),
                'scope' => $this->request->input('scope'),
                'context' => $this->request->input('context'),
                'grant_type' => 'authorization_code',
            ];
            $data = Http::post('https://login.bigcommerce.com/oauth2/token',$object);

            // Ensure token is returned successfully
            if (isset($data['access_token'])) {
                // Save installation data to the database
                $storeHash = str_replace('stores/', '', $data['context']);
                $authCode = $this->request->input('code');
                $scopes = explode(' ', $this->request->input('scope')); // Assuming scopes are space-separated
                $accessToken = $data['access_token'];
                $refreshToken = $data['refresh_token'] ?? null;
                $expiresIn = $data['expires_in'] ?? 2;
                $tokenExpiresAt = now()->addDay($expiresIn);

                    Log::info("Storing in the database ");
                    StoreInstallation::updateOrCreate(
                        ['store_hash' => $storeHash],
                        [
                            'auth_code' => $authCode,
                            'scopes' => $scopes,
                            'access_token' => $accessToken,
                            'refresh_token' => $refreshToken,
                            'token_expires_at' => $tokenExpiresAt,
                        ]
                    );

                session([
                    'store_hash' => $data['context'],
                    'access_token' => $data['access_token'],
                    'user_id' => $data['user']['id'],
                    'user_email' => $data['user']['email'],
                ]);

                // Redirect to the success page if external_install is set
                if ($this->request->has('external_install')) {
                    return Redirect::to("https://login.bigcommerce.com/app/" . $this->configuration->getAppClientId() . "/install/succeeded");
                }

            }

            return Redirect::to($this->redirectHome());

        } catch (\Exception $e) {
            // Log and redirect to the error page if there's an exception
            Log::error('Error during installation: ' . $e->getMessage());
            return redirect()->route('error', ['error_message' => $e->getMessage()]);
        }
    }

    public function load()
    {
        // Get the signed payload from the request
        $signedPayload = $this->request->input('signed_payload');
        // Ensure signed payload is not empty
        if (!empty($signedPayload)) {
            $verifiedData = $this->verifySignedRequest($signedPayload);

            if ($verifiedData !== null) {
                $storeHash = str_replace('stores/', '', $verifiedData['context']);

                $installation = StoreInstallation::findByStoreHash($storeHash);
                Log::debug("installation id: {$storeHash}");

                if (!$installation || !$installation->isValid()) {
                    return redirect()->route('error', ['error_message' => 'Store not installed or installation expired.']);
                }

                $user = User::where('email', $verifiedData['user']['email'])
                    ->where("store_id",$installation->id)
                    ->first();
                try{
                    // Check if user exists

                    if (!$user) {
                        // Create the user if not exists
                        $user = User::updateOrCreate([
                            'email' => $verifiedData['user']['email'],
                            'name' => $verifiedData['user']['email'],
                            'password' => bcrypt(Str::random(32)),
                            'store_id'=>$installation->id,
                            'platform_type'=>'bcc'
                        ]);
                    }
                }catch (\Exception $e) {
                    return redirect()->route('error', ['error_message' => $e->getMessage()]);
                }


                session()->regenerate();
                // Generate JWT for the user
                $token = JWTAuth::fromUser($user);
                Log::info('Generated token', ['token' => $token]);
                try{
                    Auth::login($user);
                }catch (\Exception $e) {
                    Log::error('Error during authentication: ' . $e->getMessage());
                }

                Log::info('User authenticated via signed_payload after login:', ['auth' => Auth::check(), 'user' => Auth::user()]);
                Log::info('Session ID', ['id' => session()->getId()]);
                return redirect()->route('dashboard.iframe', [
                    'token' => $token,
                    'store_hash' => $storeHash,
                ]);



            } else {
                return redirect()->route('error', ['error_message' => 'Invalid signed request.']);
            }
        } else {
            return redirect()->route('error', ['error_message' => 'Empty signed request.']);
        }
    }

    public function error()
    {
        // Get error message from the session and display it
        $errorMessage = session('error_message', 'Internal Application Error');
        return response("<h4>An issue has occurred:</h4><p>$errorMessage</p><a href='{$this->baseURL}'>Go back to home</a>", 400);
    }

    private function verifySignedRequest($signedRequest)
    {
        // Split the signed request into data and signature parts
        list($encodedData, $encodedSignature) = explode('.', $signedRequest, 2);

        // Decode and validate the signature
        $signature = base64_decode($encodedSignature);
        $jsonStr = base64_decode($encodedData);
        $data = json_decode($jsonStr, true);
        // Validate the signature by comparing it to the expected signature
        $expectedSignature = hash_hmac('sha256', $jsonStr, $this->configuration->getAppSecret(), false);
        if (!hash_equals($expectedSignature, $signature)) {
            Log::error('Invalid signed request from BigCommerce!');
            return null; // Return null if signature is invalid
        }

        return $data; // Return the decoded data if the signature is valid
    }

    private function redirectHome(): string
    {
        $url = config('app.env') !== 'local'
            ? '/'
            : rtrim(config('app.ngrok_url', config('app.url')), '/') . '/';

        Log::info('Redirecting to: ' . $url);

        return $url;
    }



}
