<?php

namespace Mtc\BigCommerceRates\Livewire\BigCommerce\Product;

use Mtc\BigCommerceRates\Models\BigCommerce\Product;
use Mtc\BigCommerceRates\Models\StoreInstallation;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Livewire\Component;
use Livewire\WithPagination;
use Illuminate\Pagination\LengthAwarePaginator;
use Tymon\JWTAuth\Facades\JWTAuth;


class Import extends Component
{
    use WithPagination;

    public $storeHash;


    public function mount($storeHash)
    {
        // Try to get from parameter, then from query, then from session, then from the authenticated user
        $this->storeHash = $storeHash
            ?? request()->query('storeHash')
            ?? session('storeHash')
            ?? (auth()->user()?->storeInstallation?->store_hash ?? null);
    }

    public function getUserForStore()
    {
        $storeHash = $this->storeHash;
        Log::info('Store hash:', ['store_hash' => $storeHash]);
        $cookieName = 'bc_jwt_' . $storeHash;
        $token = request()->cookie($cookieName)?? $_COOKIE[$cookieName];
        Log::info('JWT cookie value:', ['cookie' => $cookieName, 'token' => $token]);
        if ($token) {
            try {
                $user = \Tymon\JWTAuth\Facades\JWTAuth::setToken($token)->authenticate();
                Log::info('User from JWT:', ['user' => $user]);
                return $user;
            } catch (\Exception $e) {
                Log::error('JWT error:', ['message' => $e->getMessage()]);
                return null;
            }
        }
        return null;
    }

    public function importProducts()
    {
        // Get the current user
        $user = Auth::user();
        $store = $user->storeInstallation;

        if (!$store) {
            session()->flash('error', 'No store installation found for this user.');
            return;
        }
        $storeHash = $store->store_hash;
        $importedCount = $this->importProductsForStore($storeHash, $store, 1);
        session()->flash('success', "Successfully imported or updated {$importedCount} products for store: {$storeHash}");

        // Reset pagination to show the new data
        $this->resetPage();
    }

    protected function importProductsForStore($selectedStore, $store, $page = 1)
    {
        $endpointUrl = env('APP_URL')."bc-api/{$selectedStore}/v3/catalog/products";
        $queryParams = [
            "include" => "custom_fields",
            "include_fields" => "id,name,sku,price",
            "limit" => 250,
            "direction" => "desc",
            "page" => $page
        ];
        try {
            $response = Http::timeout(400)
                ->connectTimeout(1)
                ->get($endpointUrl, $queryParams)
                ->collect();
        } catch (\Illuminate\Http\Client\HttpClientException $clientException) {
            session()->flash('error', "HTTP Client Exception: " . $clientException->getMessage());
            return 0;
        }
        if ($response->isEmpty()) {
            session()->flash('error', "No products returned from API.");
            return 0;
        }
        $importedCount = 0;
        foreach ($response["data"] as $productData) {

            Product::updateOrCreate(
                [
                    'product_id' => $productData['id'],
                    'store_id' => $store->id,
                ],
                [
                    'sku' => $productData['sku'] ?? null,
                    'name' => $productData['name'] ?? '',
                    'price' => $productData['price'] ?? '0',
                    'custom_fields' => $productData['custom_fields'] ?? null,
                ]
            );
            $importedCount++;
        }
        $currentPage = data_get($response, 'meta.pagination.current_page', 1);
        $totalPages = data_get($response, 'meta.pagination.total_pages', 1);
        if ($currentPage < $totalPages) {
            $importedCount += $this->importProductsForStore($selectedStore, $store, $currentPage + 1);
        }
        return $importedCount;
    }

    public function render()
    {
        $user = $this->getUserForStore();
        $store = $user->storeInstallation;
        if ($store) {
            $products = Product::where('store_id', $store->id)
                ->latest()
                ->paginate(10);
        } else {
            // Always return a paginator, even if empty
            $products = new LengthAwarePaginator([], 0, 10);
        }

        return view('bigcommerce::livewire.bigcommerce.product.import', [
            'products' => $products
        ]);
    }
}
