<?php

namespace Mtc\BigCommerceRates\Console\Commands;

use Illuminate\Console\Command;
use Mtc\BigCommerceRates\Models\StoreInstallation;
use Illuminate\Http\Client\HttpClientException;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;
use Mtc\BigCommerceRates\Models\BigCommerce\Product;

class ImportProducts extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'bc:import-products';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Command description';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        // Fetch store_hashes
        $storeHashes = StoreInstallation::pluck('store_hash')->toArray();

        if (empty($storeHashes)) {
            $this->error('No store installations found.');
            return Command::FAILURE;
        }

        // Prompt selection
        $selectedStore = $this->choice(
            'Select the store_hash to import products for:',
            $storeHashes
        );

        $store = StoreInstallation::where('store_hash', $selectedStore)->first();
        if (!$store) {
            $this->error("Store installation not found for: {$selectedStore}");
            return Command::FAILURE;
        }


        $this->info("You selected: {$selectedStore}");


        $importedCount = $this->importProducts(
            selectedStore:  $selectedStore,
            store: $store,
            page: 1
            );

        $this->info("Successfully imported or updated {$importedCount} products for store: {$selectedStore}");
        // Here, call your product import logic
        // Example placeholder:
        $this->info("Importing products for store: {$selectedStore}");

        return Command::SUCCESS;
    }

    protected function importProducts($selectedStore,$store,?int $page=1)
    {

        $endpointUrl = env('APP_URL')."bc-api/{$selectedStore}/v3/catalog/products";

        $queryParams = [
            "include" => "custom_fields",
            "include_fields" => "id,name,sku,price",
            "limit" => 250,
            "direction" => "desc",
            "page" => $page
        ];

        try{
            $response = Http::timeout(400)
                ->connectTimeout(1)
                ->get( $endpointUrl,$queryParams)
                ->collect();

        }catch (HttpClientException $clientException){
            $this->error("HTTP Client Exception: " . $clientException->getMessage());
            return Command::FAILURE;
        }

        if ($response->isEmpty()) {
            $this->warn("No products returned from API.");
            return Command::SUCCESS;
        }

        $importedCount = 0;
        foreach ($response['data'] as $productData) {
            Product::updateOrCreate(
                [
                    'product_id' => $productData['id'],
                    'store_id' => $store->id,
                ],
                [
                    'sku' => $productData['sku'] ?? null,
                    'name' => $productData['name'] ?? '',
                    'price' => $productData['price'] ?? '0',
                    'custom_fields' => $productData['custom_fields']?? null,
                ]
            );
            $importedCount++;
        }

        $currentPage = data_get($response, 'meta.pagination.current_page', 1);
        $totalPages = data_get($response, 'meta.pagination.total_pages', 1);

        if ($currentPage <= $totalPages) {
            echo "Processing page {$currentPage} of {$totalPages}\n";
            $this->importProducts(
                selectedStore:  $selectedStore,
                store: $store,
                page: $currentPage + 1
            );
        }


        return $importedCount;
    }
}
