<?php

namespace Mtc\Blog;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Mtc\Blog\Scopes\ViewableArticle;
use Mtc\Blog\Traits\HasUrl;
use Mtc\Taxonomies\Taxonomy;

/**
 * Class Article
 *
 * @package Mtc\Blog
 */
class Article extends Model
{
    use SoftDeletes;
    use HasUrl;

    /**
     * Table Name
     *
     * @var string
     */
    protected $table = 'blog_articles';

    /**
     * Mass assignable attributes
     *
     * @var array
     */
    protected $fillable = [
        'segment',
        'title',
        'published_at',
        'status',
        'author_id',
        'is_featured',
        'image',
        'excerpt',
        'search_content',
    ];

    /**
     * Cast variables to specific types
     *
     * @var array
     */
    protected $casts = [
        'published_at' => 'datetime',
    ];

    /**
     * Boot the model
     */
    protected static function boot()
    {
        parent::boot();
        self::addGlobalScope(new ViewableArticle());

        self::creating(function (self $article) {
            $article->generateSlug();
        });
    }

    /**
     * Article versions
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function versions()
    {
        return $this->hasMany(ArticleVersion::class, 'article_id');
    }

    /**
     * Active version of articles content
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function activeVersion()
    {
        return $this->hasOne(ArticleVersion::class, 'article_id')
            ->where('is_active', 1);
    }

    /**
     * Blog tags
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
     */
    public function tags()
    {
        return $this->belongsToMany(Taxonomy::class, 'blog_article_taxonomies')
            ->wherePivot('type', '=', 'tag');
    }

    /**
     * Blog categories
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
     */
    public function categories()
    {
        return $this->belongsToMany(Taxonomy::class, 'blog_article_taxonomies')
            ->wherePivot('type', '=', 'category');
    }

    /**
     * Get the content attribute
     *
     * @return mixed
     */
    public function getContentAttribute()
    {
        return $this->activeVersion->content;
    }
}
