<?php

namespace Mtc\Checkout\Invoice;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Facades\URL;
use Mtc\Checkout\Contracts\HasDeferredCharges;
use Mtc\Checkout\Facades\Payment as PaymentProcessor;
use Mtc\Checkout\Invoice;
use Mtc\Core\Currency;
use Mtc\Refunds\Facades\Refunds;
use Mtc\Refunds\Refund;

/**
 * Class Payment
 *
 * @package Mtc\Checkout\Invoice
 */
class Payment extends Model
{
    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'invoice_payments';

    /**
     * Mass assignable attributes
     *
     * @var array
     */
    protected $fillable = [
        'provider',
        'amount',
        'reference',
        'details',
        'confirmed_at',
        'currency_code',
        'amount_in_currency',
    ];

    /**
     * Cast attributes to specific types
     *
     * @var array
     */
    protected $casts = [
        'confirmed_at' => 'datetime',
        'failed_at' => 'datetime',
        'details' => 'array',
    ];

    /**
     * Boot model
     */
    protected static function boot()
    {
        parent::boot();

        static::saving(function (self $payment) {
            if ($payment->isDirty('confirmed_at')) {
                $payment->failed_at = null;
                $payment->failure_details = null;
            }
        });
    }

    /**
     * Relationship with the invoice
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function invoice()
    {
        return $this->belongsTo(Invoice::class);
    }

    /**
     * Relationship to payment currency
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function currency(): BelongsTo
    {
        return $this->belongsTo(Currency::class, 'currency_code', 'currency');
    }

    /**
     * @return HasMany
     */
    public function refunds(): HasMany
    {
        return $this->hasMany(Refund::class, 'payment_id');
    }

    /**
     * Check payment amount that can be refunded
     *
     * @return mixed
     */
    public function getRefundableAmountAttribute()
    {
        return $this->amount - $this->refunds()->sum('amount');
    }

    /**
     * Check if payment is confirmed
     *
     * @return bool
     */
    public function getIsConfirmedAttribute()
    {
        return !empty($this->attributes['confirmed_at']);
    }

    /**
     * Check if this payment is something that can be charged as a deferred payment
     *
     * @return bool
     */
    public function allowDeferredCharge()
    {
        PaymentProcessor::setActiveDriver($this->provider);
        if (PaymentProcessor::supports(HasDeferredCharges::class) === false) {
            return false;
        }

        return PaymentProcessor::allowDeferredCharge($this);
    }

    /**
     * Check if it is possible to refund payment
     *
     * @return bool
     */
    public function isRefundable()
    {
        return config('refunds.enabled') ? Refunds::canRefundPayment($this) : false;
    }

    /**
     * Invite user to pay again due to failed payment
     *
     * @return bool
     */
    public function triggerRescue()
    {
        $subject = __('checkout::admin.recover_payment_email_subject', [
            'reference' => $this->invoice->reference,
        ]);

        $html = template('emails/invoices/recover_payment.twig', [
            'invoice' => $this->invoice,
            'recovery_url' => URL::signedRoute('view_invoice', [
                'id' => $this->invoice_id,
                'recovery' => true,
            ]),
        ]);

        return email($this->invoice->email, $subject, $html);
    }
}
