<?php

namespace Mtc\Checkout\Factories;

use Carbon\Carbon;
use Illuminate\Support\Facades\App;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Checkout\Contracts\InvoiceFactoryContract;
use Mtc\Checkout\Contracts\PayableContract;
use Mtc\Checkout\Invoice;
use Mtc\Checkout\Invoice\Address;
use Mtc\Money\Facades\Currency;

/**
 * Class InvoiceFactory
 *
 * @package Mtc\Checkout
 */
class InvoiceFactory implements InvoiceFactoryContract
{
    /**
     * Create an invoice
     *
     * @return InvoiceRepositoryContract
     */
    public function create(PayableContract $payable): InvoiceRepositoryContract
    {
        /** @var Invoice $invoice */
        $invoice = Invoice::query()
            ->create([
                'payable_type' => $payable->getPayableClass(),
                'payable_id' => $payable->getAttribute('id'),
                'due_at' => Carbon::now(),
                'amount' => $payable->getAmount(),
                'use_ex_vat' => $payable->useExVat(),
                'currency' => Currency::getCurrentCurrency(),
                'amount_in_currency' => Currency::inCurrency($payable->getAmount()),
                'email' => $payable->getAttribute('email'),
                'member_id' => $payable->getMember()->id ?? null,
                'reference' => $payable->getReference(),
                'outstanding_amount' => $payable->getAmount(),
                'vat_value' => $payable->getAttribute('vat_value'),
                'details' => $payable->getAdditionalDetails(),
                'template' => $payable->getTemplate(),
            ]);

        collect($payable->getItems())
            ->each(function ($item) use ($invoice) {
                $invoice->items()->create([
                    'purchasable_id' => $item['purchasable_id'],
                    'purchasable_type' => $item['purchasable_type'],
                    'parent_type' => get_class($item),
                    'parent_id' => $item['id'],
                    'name' => $item['name'],
                    'original_value' => $item['unit_price']->raw(true),
                    'original_value_ex_vat' => $item['unit_price_ex_vat'],
                    'value' => $item['paid_price']->raw(true),
                    'value_ex_vat' => $item['paid_price_ex_vat'],
                    'quantity' => $item['quantity'],
                    'line_total' => $item['quantity'] * $item['paid_price']->raw(true),
                    'line_total_ex_vat' => $item['quantity'] * $item['paid_price_ex_vat'],
                    'vat_rate' => $item['vat_rate'],
                    'details' => $item['attribute_fields'],
                ]);
            });

        collect($payable->getLines())
            ->each(function ($line) use ($invoice) {
                $invoice->lines()->create($line);
            });

        $address = $payable->getAddress();
        $address['type'] = Address::TYPE_RECIPIENT;
        $invoice->addresses()->create($address);
        $invoice->addresses()->create(config('invoices.seller_address'));

        /** @var InvoiceRepositoryContract $repository */
        $repository = App::make(InvoiceRepositoryContract::class);
        $repository->setModel($invoice);
        return $repository;
    }
}
