<?php

namespace Mtc\Checkout\Factories;

use Illuminate\Http\Request;
use Mtc\Checkout\Contracts\SplitInvoiceFactoryContract;
use Mtc\Checkout\Invoice;
use Mtc\Money\Facades\Currency;
use Mtc\Money\Price;

/**
 * Class SplitInvoiceFactory
 *
 * Factory that handles splitting invoice into multiple
 *
 * @package Mtc\Checkout
 */
class SplitInvoiceFactory implements SplitInvoiceFactoryContract
{
    /**
     * Create an invoice
     *
     * @return void
     */
    public function create(Request $request, Invoice $invoice)
    {
        $second_invoice = $this->createSecondInvoice($invoice);
        $this->processItemChanges($request, $invoice, $second_invoice);
        $this->processLineChanges($request, $invoice, $second_invoice);
        $invoice->save();
        $second_invoice->save();
    }

    /**
     * Create second invoice
     *
     * @param $invoice
     * @return Invoice
     */
    protected function createSecondInvoice($invoice)
    {
        $second_invoice = new Invoice();
        $second_invoice->fill($invoice->getAttributes());
        $second_invoice->amount = 0;
        $second_invoice->amount_ex_vat = 0;
        $second_invoice->amount_in_currency = 0;
        $second_invoice->vat_value = 0;
        $second_invoice->outstanding_amount = 0;
        $second_invoice->outstanding_amount_ex_vat = 0;
        $second_invoice->save();
        return $second_invoice;
    }

    /**
     * Process changes in invoice items
     *
     * @param Request $request
     * @param Invoice $invoice
     * @param Invoice $second_invoice
     */
    protected function processItemChanges(Request $request, Invoice $invoice, Invoice $second_invoice)
    {
        $invoice->items
            ->each(function ($item) use ($request, $invoice, $second_invoice) {
                if (array_key_exists($item->id, $request->input('move_item', []))) {
                    $quantity_to_move = $request->input('move_item.' . $item->id, 0);
                    if ($quantity_to_move == $item->quantity) {
                        $item->invoice_id = $second_invoice->id;
                        $item->save();
                        $this->reduceInvoiceAmount($invoice, $item->line_total)->save();
                        $this->increaseInvoiceAmount($second_invoice, $item->line_total)->save();
                    } else {
                        $item->quantity -= $quantity_to_move;
                        $item->line_total = $item->quantity * $item->value->raw(true);
                        $item->line_total_ex_vat = $item->quantity * $item->value->raw(false);
                        $item->save();
                        $new_item = new Invoice\Item;
                        $new_item->fill($item->getAttributes());
                        $new_item->invoice_id = $second_invoice->id;
                        $new_item->quantity = $quantity_to_move;
                        $new_item->value = $item->value->raw(true);
                        $new_item->line_total = $new_item->quantity * $item->value->raw(true);
                        $new_item->line_total_ex_vat = $new_item->quantity * $item->value->raw(false);
                        $new_item->save();
                        $this->reduceInvoiceAmount($invoice, $new_item->line_total)->save();
                        $this->increaseInvoiceAmount($second_invoice, $new_item->line_total)->save();
                    }
                }
            });
    }

    /**
     * Process changes in invoice lines
     *
     * @param Request $request
     * @param Invoice $invoice
     * @param Invoice $second_invoice
     */
    protected function processLineChanges(Request $request, Invoice $invoice, Invoice $second_invoice)
    {
        $invoice->lines
            ->each(function ($item) use ($request, $invoice, $second_invoice) {
                if (in_array($item->id, $request->input('move_line', []))) {
                    $item->invoice_id = $second_invoice->id;
                    $item->save();
                    $this->reduceInvoiceAmount($invoice, $item->line_total)->save();
                    $this->increaseInvoiceAmount($second_invoice, $item->line_total)->save();
                }
            });
    }

    /**
     * Reduce the amount of an invoice by given amount
     *
     * @param $invoice
     * @param $amount
     * @param $use_ex_vat
     * @return mixed
     */
    protected function reduceInvoiceAmount($invoice, Price $amount)
    {
        $invoice->amount -= $amount->withTax();
        $invoice->amount_ex_vat -= $amount->withoutTax();
        $invoice->amount_in_currency -= Currency::inBaseCurrency($amount->withTax(), $invoice->currency);
        $invoice->outstanding_amount -= $amount->withTax();
        $invoice->outstanding_amount_ex_vat -= $amount->withoutTax();
        $invoice->vat_value -= $amount->tax();
        return $invoice;
    }

    /**
     * Reduce the amount of an invoice by given amount
     *
     * @param $invoice
     * @param $amount
     * @param $use_ex_vat
     * @return mixed
     */
    protected function increaseInvoiceAmount($invoice, Price $amount)
    {
        $invoice->amount += $amount->withTax();
        $invoice->amount_ex_vat += $amount->withoutTax();
        $invoice->amount_in_currency += Currency::inBaseCurrency($amount->withTax(), $invoice->currency);
        $invoice->outstanding_amount += $amount->withTax();
        $invoice->outstanding_amount_ex_vat += $amount->withoutTax();
        $invoice->vat_value += $amount->tax();
        return $invoice;
    }
}
