<?php

namespace Mtc\Checkout\Providers;

use Illuminate\Support\Facades\Event;
use Illuminate\Support\ServiceProvider;
use Mtc\Checkout\Console\Commands\InstallCheckout;
use Mtc\Checkout\Contracts\InvoiceFactoryContract;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Checkout\Contracts\PayableFactoryContract;
use Mtc\Checkout\Contracts\PaymentResponseContract;
use Mtc\Checkout\Contracts\SplitInvoiceFactoryContract;
use Mtc\Checkout\Events\AddPaymentGatewayScripts;
use Mtc\Checkout\Facades\Payment;
use Mtc\Checkout\PaymentManager;
use Mtc\Checkout\ZeroAmountOrderPayment;

/**
 * Class CheckoutServiceProvider
 *
 * @package Mtc\Checkout\Providers
 */
class CheckoutServiceProvider extends ServiceProvider
{
    /**
     * Bootstrap the application services.
     */
    public function boot()
    {
        /*
         * Optional methods to load your package assets
         */
         $this->loadMigrationsFrom(dirname(__DIR__, 2) . '/database/migrations');
         $this->loadTranslationsFrom(dirname(__DIR__, 2) . '/resources/lang', 'checkout');

        if ($this->app->runningInConsole()) {
            $this->publishes([
                dirname(__DIR__, 2) . '/config/checkout.php' => config_path('checkout.php'),
            ], 'config');
            $this->publishes([
                dirname(__DIR__, 2) . '/config/invoices.php' => config_path('invoices.php'),
            ], 'config');

            $this->publishes([
                \dirname(__DIR__, 2) . '/resources/css' => public_path('css'),
            ], 'assets');

            // Publishing assets.
            $this->publishes([
                dirname(__DIR__, 2) . '/resources/js' => resource_path('js/vendor'),
            ], 'assets');

            $this->commands([
                InstallCheckout::class,
            ]);

        } else {
            app('twig_loader')->addPath(dirname(__DIR__, 2) . '/resources/templates');
            app('twig')->addFunction(new \Twig_SimpleFunction('addPaymentGatewayScripts', function () {
                return Event::dispatch(new AddPaymentGatewayScripts());
            }));

            app('twig')->addFunction(new \Twig_SimpleFunction('externalPayments', function () {
                return Payment::externalGateways();
            }));

            $this->app['payment']->extend('zero-amount-payment', function ($app) {
                return new ZeroAmountOrderPayment();
            });

        }
    }

    /**
     * Register the application services.
     */
    public function register()
    {
        // Automatically apply the package configuration
        $this->mergeConfigFrom(dirname(__DIR__, 2) . '/config/checkout.php', 'checkout');
        $this->mergeConfigFrom(dirname(__DIR__, 2) . '/config/invoices.php', 'invoices');

        collect(config('checkout.service_providers', []))
            ->each(function ($service_provider) {
                $this->app->register($service_provider);
            });

        $this->app->bind(PayableFactoryContract::class, config('checkout.payable_factory'));
        $this->app->bind(InvoiceFactoryContract::class, config('checkout.invoice_factory'));
        $this->app->bind(SplitInvoiceFactoryContract::class, config('invoices.split_invoice_factory'));
        $this->app->bind(InvoiceRepositoryContract::class, config('checkout.invoice_repository'));
        $this->app->bind(PaymentResponseContract::class, config('checkout.payment_response_handler'));

        // Register the main class to use with the facade
        $this->app->singleton('payment', function ($app) {
            return new PaymentManager($app);
        });
    }

}
