<?php

namespace Mtc\Checkout\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Event;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\URL;
use Mtc\Basket\Contracts\BasketRepositoryInterface;
use Mtc\Checkout\Contracts\InvoiceFactoryContract;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Checkout\Contracts\PayableFactoryContract;
use Mtc\Checkout\Events\PaymentSuccessPage;
use Mtc\Checkout\Facades\Payment;
use \Mtc\Checkout\Invoice\Payment as PaymentModel;
use Mtc\Money\Facades\Currency;

/**
 * Class PaymentController
 *
 * @package Mtc\Checkout
 */
class PaymentController extends Controller
{
    /**
     * View and allow paying for invoice
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function show(Request $request, InvoiceRepositoryContract $invoice_repository)
    {
        // Make sure signature is tested here as we're about to view invoice
        if (!$request->hasValidSignature()) {
            return redirect()->to('/');
        }

        $invoice_repository->load($request->input('id'));
        $this->page_meta['title'] = __('checkout::admin.pay_page_title', [
            'reference' => $invoice_repository->getReference()
        ]);

        return template('invoices/show.twig', [
            'invoice' => $invoice_repository->getModel(),
            'page_meta' => $this->page_meta,
        ]);
    }

    /**
     * Create an invoice and load applicable payment gateways for requested payment
     *
     * We send in the type of request we want to process. This can be basket, subscription, booking, one-off invoice
     * Payable Factory will consume the request and return a payable object based on the passed request
     * e.g. for basket this would be order, subscriptions might return the subscription object itself
     * Then an invoice is generated based on payable. This way we always make payment on invoice and have decoupled
     * object that is receiving payment.
     * Lastly we run the invoice and payable through registered payment providers to provide applicable payment options
     * as some might be filtered out due to restrictions
     *
     * @param Request $request
     * @param PayableFactoryContract $payable_factory
     * @param InvoiceFactoryContract $invoice_factory
     * @return mixed
     */
    public function initializePayment(Request $request, PayableFactoryContract $payable_factory, InvoiceFactoryContract $invoice_factory)
    {
        if (!App::make(BasketRepositoryInterface::class)->getModel()->exists()) {
            throw new \Exception('Basket does not exist');
        }
        $payable = $payable_factory->create($request);
        $invoice = $invoice_factory->create($payable);

        return Payment::getApplicablePaymentForms($invoice, $payable);
    }

    /**
     * Initialize payment of an already existing invoice
     * 
     * @param Request $request
     * @param $invoice_id
     * @param InvoiceRepositoryContract $invoice_repository
     * @return mixed
     */
    public function initializeInvoice(Request $request, $invoice_id, InvoiceRepositoryContract $invoice_repository)
    {
        $invoice_repository->load($invoice_id);
        return Payment::getApplicablePaymentForms($invoice_repository);
    }

    /**
     * Take charge on payment
     *
     * This is the default way of charging an invoice
     * It is recommended to use this flow for taking payment on linear payment gateways ( overview => payment => success/failure )
     *
     * @param Request $request
     * @param $invoice_id
     * @param InvoiceRepositoryContract $invoice_repository
     * @return \Illuminate\Http\RedirectResponse
     */
    public function charge(Request $request, $invoice_id, InvoiceRepositoryContract $invoice_repository)
    {
        $invoice_repository->load($invoice_id);
        Payment::setActiveDriver($request->input('gateway'));

        // Only try charging if order is not paid
        if ($invoice_repository->isPaid()) {
            return $this->checkAndRedirectPaidInvoice($request, $invoice_repository);
        }

        try {
            $payment_details = Payment::charge($request, $invoice_repository);
            $invoice_repository->savePayment($payment_details);
            return redirect()->to(URL::signedRoute('successful_payment', ['id' => $invoice_id ], null, config('app.env') === 'production'));
        } catch (\Exception $exception) {
            Log::warning('Failed payment charge', [
                'invoice_id' => $invoice_id,
                'exception' => $exception,
                'request' => $request->all()
            ]);
            return redirect()->to('/payment/failed');
        }
    }

    /**
     * Create a deferred charge on a payment
     *
     * @param Request $request
     * @param $invoice_id
     * @param PaymentModel $payment
     * @return \Illuminate\Http\RedirectResponse
     */
    public function deferredCharge(Request $request, $invoice_id, PaymentModel $payment, InvoiceRepositoryContract $invoice_repository)
    {
        try {
            $reduce_amount = $payment->amount;
            if ($request->input('amount') > 0 && $payment->amount > $request->input('amount')) {
                $payment->amount_in_currency = $request->input('amount');
                $payment->amount = Currency::inBaseCurrency($payment->amount_in_currency, $payment->currency_code);
            } elseif ($request->input('amount') <= 0) {
                throw new \Exception('Charge amount must be between 0 and payment value');
            }

            Payment::setActiveDriver($payment->provider);
            Payment::chargeDeferredPayment($payment);

            $payment->confirmed_at = now();
            $payment->save();

            $invoice_repository->load($invoice_id);
            $invoice_repository->reduceOutstandingAmount($reduce_amount);
            $invoice_repository->checkIfInvoiceIsPaid();
        } catch (\Exception $exception) {
            $payment->failed_at = now();
            $payment->failure_details = $exception->getMessage();
            $payment->save();
            Log::warning('Failed deferred charge', [
                'payment' => $payment,
                'exception' => $exception,
            ]);
            session()->flash('error', 'Charge failed: ' . $exception->getMessage());
        }
        return redirect()->back();
    }

    /**
     * Render payment successful page
     *
     * This uses a signed route to
     *
     * @param Request $request
     * @param InvoiceRepositoryContract $invoice_repository
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function success(Request $request, InvoiceRepositoryContract $invoice_repository)
    {
        if (!$request->hasValidSignature(config('app.env') === 'production')) {
            return redirect()->to('/');
        }

        $invoice_repository->load($request->input('id'));
        $success_events = collect(Event::dispatch(new PaymentSuccessPage($request, $invoice_repository)))
            ->filter();

        return App::make(config('checkout.payment_response_handler'))
            ->success($invoice_repository, $success_events);
    }

    /**
     * Render payment failed page
     *
     * @param InvoiceRepositoryContract $invoice_repository
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function failed(InvoiceRepositoryContract $invoice_repository)
    {
        return App::make(config('checkout.payment_response_handler'))->failure($invoice_repository);
    }

    /**
     * Deal with an already paid invoice
     *
     * @param Request $request
     * @param InvoiceRepositoryContract $invoice
     * @return \Illuminate\Http\RedirectResponse
     */
    protected function checkAndRedirectPaidInvoice(Request $request, InvoiceRepositoryContract $invoice)
    {
        // Allow viewing success page if basket matches session
        if (App::make(config('checkout.verify_user_viewing_paid_order'))->verify($request, $invoice)) {
            return redirect()->to(URL::signedRoute('successful_payment', ['id' => $invoice->getId() ], null, config('app.env') === 'production'));
        }

        return redirect()->to('/');
    }

}

