<?php

namespace Mtc\Checkout;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\URL;
use Mtc\Checkout\Invoice\Address;
use Mtc\Checkout\Invoice\Item;
use Mtc\Checkout\Invoice\Line;
use Mtc\Checkout\Invoice\Payment;
use Mtc\Modules\Members\Classes\Member;
use Mtc\Money\HasPrices;
use Mtc\Orders\Order;

/**
 * Class Invoice
 *
 * @package Mtc\Checkout
 */
class Invoice extends Model
{
    use HasPrices;

    /**
     * Mass assignable variables
     *
     * @var array
     */
    protected $fillable = [
        'payable_id',
        'payable_type',
        'reference',
        'amount',
        'use_ex_vat',
        'amount_ex_vat',
        'amount_in_currency',
        'currency',
        'email',
        'contact_number',
        'outstanding_amount',
        'outstanding_amount_ex_vat',
        'vat_value',
        'details',
        'due_at',
        'template',
        'member_id',
        'invite_emails_sent',
        'last_payment_invite_email',
    ];

    /**
     * Cast variables to specific types
     *
     * @var array
     */
    protected $casts = [
        'due_at' => 'datetime',
        'last_payment_invite_email' => 'datetime',
        'details' => 'array',
    ];

    /**
     * Boot model
     */
    protected static function boot()
    {
        parent::boot();

        self::retrieved(function (self $invoice) {
            if ($invoice->attributes['use_ex_vat']) {
                $invoice->load_ex_vat_price = true;
            }
        });

        self::deleted(function (self $invoice) {
            $invoice->items()->delete();
            $invoice->lines()->delete();
            $invoice->addresses()->delete();
            $invoice->payments()->delete();
        });
    }

    /**
     * Relationship with invoice items
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function items()
    {
        return $this->hasMany(Item::class);
    }

    /**
     * Relationship with invoice lines
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function lines()
    {
        return $this->hasMany(Line::class);
    }

    /**
     * Relationship with invoice lines
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function delivery()
    {
        return $this->hasOne(Line::class)->where('type', 'delivery');
    }

    /**
     * Relationship with invoice addresses
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function addresses()
    {
        return $this->hasMany(Address::class);
    }

    /**
     * Relationship with invoice addresses
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function recipient()
    {
        return $this->hasOne(Address::class)->where('type', Address::TYPE_RECIPIENT);
    }

    /**
     * Relationship with payment on invoice
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function payments()
    {
        return $this->hasMany(Payment::class);
    }

    /**
     * Relationship with payment on invoice
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function member()
    {
        return $this->belongsTo(Member::class);
    }

    /**
     * Morph to invoice owner
     *
     * @return \Illuminate\Database\Eloquent\Relations\MorphTo
     */
    public function payable()
    {
        return $this->morphTo();
    }

    /**
     * Direct link to order
     * Used for some fallback situations like filtering by unpaid Orders
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function order()
    {
        return $this->belongsTo(Order::class, 'payable_id')
            ->where('invoices.payable_type', 'order');
    }

    /**
     * @param Request $request
     * @return Builder
     */
    public function scopeSearch(Builder $query, Request $request)
    {
        collect(config('invoices.admin_search_filters', []))
            ->each(function($filter_class) use ($request, $query) {
                App::make($filter_class)->handle($request, $query);
            });

        return $query;
    }


    /**
     * Check if invoice is fully paid
     *
     * @return bool
     */
    public function getIsFullyPaidAttribute()
    {
        return $this->attributes['outstanding_amount'] == 0;
    }

    /**
     * @return mixed
     */
    public function getSubtotalAttribute()
    {
        return $this->items
            ->map(function ($item) {
                return $item->line_total->raw($this->use_ex_vat == false);
            })
            ->sum();
    }

    /**
     * @return string
     */
    public function getPaymentUrlAttribute()
    {
        return URL::signedRoute('view_invoice', [
            'id' => $this->id
        ]);
    }

    /**
     * Get the verbose value of invoice
     *
     * @return array|string|null
     */
    public function getStatusNameAttribute()
    {
        return __("checkout::invoices.status.{$this->status}");
    }

    /**
     * Check if invoice is pending
     *
     * @return bool
     */
    public function isPending()
    {
        return $this->status === InvoiceStatus::PENDING;
    }

    /**
     * Check if invoice is Paid
     *
     * @return bool
     */
    public function isPaid()
    {
        return $this->status === InvoiceStatus::PAID;
    }

    /**
     * Check if invoice is cancelled
     *
     * @return bool
     */
    public function isCancelled()
    {
        return $this->status === InvoiceStatus::VOIDED;
    }
}
