<?php

namespace Mtc\Checkout\Invoice;

use Illuminate\Database\Eloquent\Model;
use Mtc\Checkout\Facades\Payment as PaymentProcessor;
use Mtc\Checkout\Invoice;
use Mtc\Core\Currency;

/**
 * Class Payment
 *
 * @package Mtc\Checkout\Invoice
 */
class Payment extends Model
{
    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'invoice_payments';

    /**
     * Mass assignable attributes
     *
     * @var array
     */
    protected $fillable = [
        'provider',
        'amount',
        'reference',
        'details',
        'confirmed_at',
        'currency_code',
        'amount_in_currency',
    ];

    /**
     * Cast attributes to specific types
     *
     * @var array
     */
    protected $casts = [
        'confirmed_at' => 'datetime',
        'failed_at' => 'datetime',
        'details' => 'array',
    ];

    /**
     * Boot model
     */
    protected static function boot()
    {
        parent::boot();

        static::saving(function (self $payment) {
            if ($payment->isDirty('confirmed_at')) {
                $payment->failed_at = null;
                $payment->failure_details = null;
            }
        });
    }

    /**
     * Relationship with the invoice
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function invoice()
    {
        return $this->belongsTo(Invoice::class);
    }

    /**
     * Relationship to payment currency
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function currency()
    {
        return $this->belongsTo(Currency::class, 'currency_code', 'currency');
    }

    /**
     * Check if payment is confirmed
     *
     * @return bool
     */
    public function getIsConfirmedAttribute()
    {
        return !empty($this->attributes['confirmed_at']);
    }

    /**
     * Check if this payment is something that can be charged as a deferred payment
     *
     * @return bool
     */
    public function allowDeferredCharge()
    {
        try {
            PaymentProcessor::setActiveDriver($this->provider);
            return PaymentProcessor::allowDeferredCharge($this);
        } catch (\Exception $exception) {
            // e.g. Payment Processor does not have this functionality
        }
        return false;
    }
}
