<?php

namespace Mtc\Checkout\Factories;

use Carbon\Carbon;
use Illuminate\Support\Facades\App;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Checkout\Contracts\InvoiceFactoryContract;
use Mtc\Checkout\Contracts\PayableContract;
use Mtc\Checkout\Invoice\Address;
use Mtc\Money\Facades\Currency;
use Mtc\Checkout\Invoice;

/**
 * Class InvoiceFactory
 *
 * @package Mtc\Checkout
 */
class BackorderInvoiceFactory implements InvoiceFactoryContract
{
    /**
     * Create an invoice
     *
     * @return InvoiceRepositoryContract
     */
    public function create(PayableContract $payable): InvoiceRepositoryContract
    {
        $backorder_value_percentage = $payable->getAttribute('backorder_cost_total') / $payable->getAmount();
        $invoice = $this->createImmediateInvoice($payable, $backorder_value_percentage);
        $this->createBackOrderInvoice($payable, $backorder_value_percentage);
        /** @var InvoiceRepositoryContract $repository */
        $repository = App::make(InvoiceRepositoryContract::class);
        $repository->setModel($invoice);
        return $repository;
    }

    /**
     * Invoice for immediate order
     *
     * @param PayableContract $payable
     * @return Invoice
     */
    protected function createImmediateInvoice(PayableContract $payable, $backorder_value_percentage)
    {
        /** @var Invoice $invoice */
        $invoice = Invoice::query()
            ->create([
                'payable_type' => $payable->getPayableClass(),
                'payable_id' => $payable->getAttribute('id'),
                'due_at' => Carbon::now(),
                'amount' => $payable->getAmount() - $payable->getAttribute('backorder_cost_total'),
                'use_ex_vat' => $payable->useExVat(),
                'currency' => Currency::getCurrentCurrency(),
                'amount_in_currency' => Currency::inCurrency($payable->getAmount() - $payable->getAttribute('backorder_cost_total')),
                'email' => $payable->getAttribute('email'),
                'contact_number' => $payable->getAttribute('contact_number'),
                'member_id' => $payable->getMember()->id ?? null,
                'reference' => $payable->getReference(),
                'outstanding_amount' => $payable->getAmount() - $payable->getAttribute('backorder_cost_total'),
                'vat_value' => $payable->getAttribute('vat_value') * (1 - $backorder_value_percentage),
                'details' => $payable->getAdditionalDetails(),
                'template' => $payable->getTemplate(),
            ]);

        collect($payable->getItems())
            ->each(function ($item) use ($invoice) {
                if (($item['quantity'] - $item['quantity_on_backorder']) <= 0) {
                    return;
                }
                $invoice->items()->create([
                    'purchasable_id' => $item['purchasable_id'],
                    'purchasable_type' => $item['purchasable_type'],
                    'parent_type' => get_class($item),
                    'parent_id' => $item['id'],
                    'name' => $item['name'],
                    'original_value' => $item['unit_price']->raw(true),
                    'original_value_ex_vat' => $item['unit_price_ex_vat'],
                    'value' => $item['paid_price']->raw(true),
                    'value_ex_vat' => $item['paid_price_ex_vat'],
                    'quantity' => ($item['quantity'] - $item['quantity_on_backorder']),
                    'line_total' => ($item['quantity'] - $item['quantity_on_backorder']) * $item['paid_price']->raw(true),
                    'line_total_ex_vat' => ($item['quantity'] - $item['quantity_on_backorder']) * $item['paid_price_ex_vat'],
                    'vat_rate' => $item['vat_rate'],
                    'details' => $item['attribute_fields'],
                ]);
            });

        collect($payable->getLines())
            ->each(function ($line) use ($invoice, $backorder_value_percentage) {
                if ($line['type'] !== 'delivery') {
                    $line['value'] *= (1 - $backorder_value_percentage);
                }
                $invoice->lines()->create($line);
            });

        $address = $payable->getAddress();
        $address['type'] = Address::TYPE_RECIPIENT;
        $invoice->addresses()->create($address);
        $invoice->addresses()->create(config('invoices.seller_address'));

        return $invoice;
    }

    /**
     * Invoice for backorder
     *
     * @param PayableContract $payable
     */
    protected function createBackOrderInvoice(PayableContract $payable, $backorder_value_percentage)
    {
        if ($payable->getAttribute('backorder_cost_total') == 0) {
            return;
        }

        /** @var Invoice $invoice */
        $invoice = Invoice::query()
            ->create([
                'payable_type' => $payable->getPayableClass(),
                'payable_id' => $payable->getAttribute('id'),
                'due_at' => Carbon::now(),
                'amount' => $payable->getAttribute('backorder_cost_total'),
                'use_ex_vat' => $payable->useExVat(),
                'currency' => Currency::getCurrentCurrency(),
                'amount_in_currency' => Currency::inCurrency($payable->getAttribute('backorder_cost_total')),
                'email' => $payable->getAttribute('email'),
                'contact_number' => $payable->getAttribute('contact_number'),
                'member_id' => $payable->getMember()->id ?? null,
                'reference' => $payable->getReference(),
                'outstanding_amount' => $payable->getAttribute('backorder_cost_total'),
                'vat_value' => $payable->getAttribute('vat_value') * $backorder_value_percentage,
                'details' => $payable->getAdditionalDetails(),
                'template' => $payable->getTemplate(),
            ]);

        collect($payable->getItems())
            ->each(function ($item) use ($invoice) {
                if ($item['quantity_on_backorder'] == 0) {
                    return;
                }
                $invoice->items()->create([
                    'purchasable_id' => $item['purchasable_id'],
                    'purchasable_type' => $item['purchasable_type'],
                    'parent_type' => get_class($item),
                    'parent_id' => $item['id'],
                    'name' => $item['name'],
                    'original_value' => $item['unit_price']->raw(true),
                    'original_value_ex_vat' => $item['unit_price_ex_vat'],
                    'value' => $item['paid_price']->raw(true),
                    'value_ex_vat' => $item['paid_price_ex_vat'],
                    'quantity' => $item['quantity_on_backorder'],
                    'line_total' => $item['quantity_on_backorder'] * $item['paid_price']->raw(true),
                    'line_total_ex_vat' => $item['quantity_on_backorder'] * $item['paid_price_ex_vat'],
                    'vat_rate' => $item['vat_rate'],
                    'details' => $item['attribute_fields'],
                ]);
            });

        collect($payable->getLines())
            ->reject(function ($line) {
                return $line['type'] === 'delivery';
            })
            ->each(function ($line) use ($invoice, $backorder_value_percentage) {
                $line['value'] *= $backorder_value_percentage;
                $invoice->lines()->create($line);
            });
        $address = $payable->getAddress();
        $address['type'] = Address::TYPE_RECIPIENT;
        $invoice->addresses()->create($address);
        $invoice->addresses()->create(config('invoices.seller_address'));
    }
}
