<?php

namespace Mtc\Checkout;

use Illuminate\Http\Request;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Checkout\Contracts\PayableContract;
use Mtc\Checkout\Contracts\PaymentGateway;
use Mtc\Checkout\PaymentForm;
use Mtc\Money\Facades\Currency;

/**
 * Zero Value payment processor.
 *
 * This is a dummy payment gateway for situations where no (immediate) payment is taken
 * The necessity for this is coming from fully discounted orders or backorder placement.
 *
 * @package  Mtc\ZeroAmountOrderPayment
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class ZeroAmountOrderPayment implements PaymentGateway
{
    /**
     * Check if the gateway is available for use on this payment.
     *
     * @param InvoiceRepositoryContract $invoice
     * @param PayableContract $payable
     * @return bool
     */
    public function isApplicable(InvoiceRepositoryContract $invoice, $payable): bool
    {
        return $invoice->getOutstandingAmount() < 0.01;
    }

    /**
     * Render the payment template.
     *
     * @param InvoiceRepositoryContract $invoice
     * @param PayableContract $payable
     * @return string
     */
    public function getPaymentForm(InvoiceRepositoryContract $invoice): PaymentForm
    {
        $invoice = template('checkout/zero_amount_payment_form.twig', [
            'invoice' => $invoice
        ]);

        return new PaymentForm($invoice, 'template', [
            'name' => __('checkout::invoice.zero_amount_payment')
        ]);
    }

    /**
     * Charge payment on invoice
     *
     * @param Request $request
     * @param InvoiceRepositoryContract $invoice
     * @return bool
     * @throws \Exception
     */
    public function charge(Request $request, InvoiceRepositoryContract $invoice): array
    {
        $data = [
            'provider' => 'zero-amount-payment',
            'amount' => $invoice->getOutstandingAmount(),
            'reference' => config('checkout.zero_amount_payment_reference_prefix') . $invoice->getId(),
            'details' => [],
            'confirmed_at' => now(),
        ];

        if (config('checkout.multi_currency_payments')) {
            $data['currency_code'] = $invoice->getCurrency();
            $data['amount_in_currency'] = Currency::inCurrency($invoice->getOutstandingAmount(), $invoice->getCurrency());
        }

        return $data;
    }

}
