<?php

namespace Mtc\Checkout\Invoice;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Collection;
use Mtc\Checkout\Invoice;
use Mtc\MercuryDataModels\Country;

/**
 * Class Address
 *
 * @package Mtc\Checkout\Invoice
 */
class Address extends Model
{
    /**
     * Invoice address types
     */
    public const TYPE_RECIPIENT = 'recipient';
    public const TYPE_SELLER = 'seller';

    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'invoice_addresses';

    /**
     * Address type constants
     */
    public const TYPE_SHIPPING = 'shipping';
    public const TYPE_BILLING = 'billing';
    public const TYPE_COLLECTION = 'collection';

    /**
     * Columns that cannot be mass assigned
     *
     * @var array
     */
    protected $guarded = [
        'id',
        'owner_country',
    ];

    /**
     * Address blueprint - empty address array
     *
     * @var array
     */
    public static $blueprint = [
        'type' => '',
        'first_name' => '',
        'last_name' => '',
        'address1' => '',
        'address2' => '',
        'city' => '',
        'county' => '',
        'state' => '',
        'postcode' => '',
        'country' => '',
    ];

    /**
     * Fields that identify address (excluding name)
     *
     * @var array
     */
    public static $address_fields = [
        'address1',
        'address2',
        'city',
        'county',
        'state',
        'postcode',
        'country',
    ];

    public function invoice(): BelongsTo
    {
        return $this->belongsTo(Invoice::class);
    }

    public function ownerCountry(): BelongsTo
    {
        return $this->belongsTo(Country::class, 'country', 'code');
    }

    public function getFullAddressAttribute(): Collection
    {
        return collect($this->only(self::$address_fields))->filter()
            ->map(function ($value, $key) {
                return $key === 'country' ? $this->formatCountryAttributeValue() : $value;
            })
            ->reject(function ($value) {
                return $value === 'NA';
            });
    }

    private function formatCountryAttributeValue(): string
    {
        return config('foundation.load_full_country_names') === true && !empty($this->ownerCountry->name) ?
            $this->ownerCountry->name : $this->country;
    }

    public function getFormattedAddressAttribute(): string
    {
        return $this->getFullAddressAttribute()->implode(', ');
    }

    public function getCustomerNameAttribute(): string
    {
        return $this->first_name . ' ' . $this->last_name;
    }
}
