<?php

namespace Mtc\Checkout;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Event;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Checkout\Events\PaidInvoice;
use Mtc\Members\Member;
use Mtc\Money\Facades\Currency;

/**
 * Class InvoiceFactory
 *
 * @package Mtc\Checkout
 */
class InvoiceRepository implements InvoiceRepositoryContract
{
    /**
     * @var Invoice $invoice
     */
    protected $invoice;

    /**
     * Set model on repository
     *
     * @param $model
     * @return void
     */
    public function setModel($model)
    {
        $this->invoice = $model;
    }

    /**
     * Get model on repository
     *
     * @param $model
     * @return Model
     */
    public function getModel()
    {
        return $this->invoice;
    }

    /**
     * Get Invoice Id
     *
     * @return int
     */
    public function getId(): ?int
    {
        return $this->invoice?->id;
    }

    /**
     * Get Invoice Reference
     *
     * @return string
     */
    public function getReference(): string
    {
        if (config('invoices.use_id_for_reference')) {
            return $this->invoice->id;
        }
        return $this->invoice->reference;
    }

    /**
     * Load invoice from Id
     *
     * @return Model
     */
    public function load($id)
    {
        $this->invoice = Invoice::query()->find($id);
        return $this->invoice;
    }

    /**
     * Get the Record object for this invoice
     *
     * @return mixed
     */
    public function getPayable()
    {
        return $this->invoice->payable;
    }

    /**
     * Return the outstanding payment amount
     *
     * @return float
     */
    public function getOutstandingAmount(): float
    {
        return $this->invoice->outstanding_amount;
    }

    public function getEmail(): ?string
    {
        return $this->invoice->email;
    }

    /**
     * Save the payment details for invoice
     *
     * @param $payment_details
     * @return void
     */
    public function savePayment($payment_details)
    {
        $this->invoice
            ->payments()
            ->updateOrCreate([
                'provider' => $payment_details['provider'],
                'reference' => $payment_details['reference'],
            ], $payment_details);

        if (!empty($payment_details['confirmed_at'])) {
            $this->reduceOutstandingAmount($payment_details['amount']);
        }

        $this->checkIfInvoiceIsPaid();

        Event::dispatch(new PaidInvoice($this, $payment_details['reference'], $payment_details));
    }

    /**
     * Check if invoice has no outstanding amount
     * Mark invoice as paid if applicable
     */
    public function checkIfInvoiceIsPaid()
    {
        if ($this->invoice->outstanding_amount <= 0) {
            $this->invoice->status = InvoiceStatus::PAID;
            $this->invoice->save();
        }
    }

    /**
     * Reduce the outstanding amount on invoice
     *
     * @param $amount
     */
    public function reduceOutstandingAmount($amount)
    {
        $this->invoice->outstanding_amount -= $amount;
        $this->invoice->save();
    }

    /**
     * Return the outstanding payment amount
     *
     * @return float
     */
    public function getOutstandingAmountInCurrency(): float
    {
        return $this->invoice->outstanding_amount;
    }

    /**
     * Get the currency for invoice
     *
     * @return string
     */
    public function getCurrency(): string
    {
        if (empty($this->invoice->currency)) {
            return Currency::getCurrentCurrency();
        }
        return $this->invoice->currency;
    }

    /**
     * Get Invoice assigned Member
     *
     * @return mixed
     */
    public function getMember()
    {
        return $this->invoice->member;
    }

    /**
     * Check if invoice has already been paid and processed
     *
     * @return bool
     */
    public function isPaid(): bool
    {
        return $this->getOutstandingAmount() == 0
            && $this->invoice->status == InvoiceStatus::PAID;
    }

    /**
     * Invite invoice owner to pay for invoice
     *
     * @return bool
     */
    public function inviteToPay(): bool
    {
        $subject = __('checkout::admin.invite_email_subject', [
            'reference' => $this->invoice->reference,
        ]);

        $html = template('emails/invoices/invite_to_pay.twig', [
            'invoice' => $this->invoice,
        ]);

        $this->invoice->invite_emails_sent++;
        $this->invoice->last_payment_invite_email = now();
        $this->invoice->save();

        return email($this->invoice->email, $subject, $html);
    }

    /**
     * Whether invoice owner qualifies for registration
     * Invoice by guest user && invoice email isn't already attached to a member
     *
     * @return bool
     */
    public function shouldOfferRegistrationOnSuccess(): bool
    {
        return $this->invoice->customer_id === null
            && Customer::query()->where('email', $this->invoice->email)->exists() === false;
    }

    /**
     * Whether the invoice has conflicts, preventing settlement
     *
     * @return bool
     */
    public function hasConflicts(): bool
    {
        $this->conflicts = [];

        $all_in_stock = collect($this->getModel()->items)
            ->filter(function ($item) {
                $item->in_stock = $item->purchasable->getStock() >= $item->quantity;
                return !$item->in_stock;
            })
            ->count() === 0;

        if (!$all_in_stock) {
            $this->conflicts[] = __('basket::basket.stock_error');
            return true;
        }

        return false;
    }

    /**
     * Get conflicts that prevent settlement
     *
     * @return array
     */
    public function getConflicts(): array
    {
        if (is_null($this->conflicts)) {
            $this->hasConflicts();
        }

        return $this->conflicts;
    }
}
