<?php

namespace Mtc\ContentManager\Http\Controllers;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Foundation\Bus\DispatchesJobs;
use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;
use Mtc\ContentManager\Http\Requests\StoreImageSizeRequest;
use Mtc\ContentManager\Http\Requests\UpdateImageSizeRequest;
use Mtc\ContentManager\Http\Resources\MediaSizeList;
use Mtc\ContentManager\Jobs\ChangeMediaSizeDimensions;
use Mtc\ContentManager\Jobs\RemoveMediaUseOutdatedSizes;
use Mtc\ContentManager\Models\MediaSize;

class MediaSizeController
{
    use ValidatesRequests;
    use DispatchesJobs;

    public function index(): MediaSizeList
    {
        return new MediaSizeList(MediaSize::all());
    }

    /**
     * Store sizes from array
     *
     * @param StoreImageSizeRequest $request
     * @return MediaSize|Model
     */
    public function store(StoreImageSizeRequest $request): MediaSize
    {
        return MediaSize::query()->updateOrCreate([
            'label' => $request->input('title'),
            'model' => $request->input('model'),
        ]);
    }

    public function show(MediaSize $mediaSize): MediaSize
    {
        return $mediaSize;
    }

    public function update(UpdateImageSizeRequest $request, MediaSize $mediaSize): MediaSize
    {
        if (!empty($mediaSize->model) && $request->input('model') !== $mediaSize->model) {
            throw ValidationException::withMessages([
                'model' => 'Model on size cannot be changed',
            ]);
        }
        if ($request->isPredefinedSize()) {
            throw ValidationException::withMessages([
                'model' => 'Dimensions not allowed as they match a predefined size',
            ]);
        }
        $originalValues = $mediaSize->getOriginal();
        $mediaSize->update($request->validated());
        $this->dispatch(new ChangeMediaSizeDimensions($mediaSize, $originalValues));
        return $mediaSize;
    }

    public function destroy(MediaSize $mediaSize): MediaSizeList
    {
        $this->dispatch(new RemoveMediaUseOutdatedSizes($mediaSize->toArray()));
        $mediaSize->delete();
        return new MediaSizeList(MediaSize::all());
    }

    /**
     * Show sizes that exist for given dimensions
     *
     * @param Request $request
     * @return \Illuminate\Database\Eloquent\Builder[]|\Illuminate\Database\Eloquent\Collection
     */
    public function forDimensions(Request $request)
    {
        $query = MediaSize::query();
        collect($request->input('sizes'))
            ->map(function ($size) {
                $dimensions = explode('x', $size);
                return [
                    'width' => $dimensions[0],
                    'height' => $dimensions[1],
                ];
            })
            ->each(function ($dimensions) use ($query) {
                $query->orWhere(function ($query) use ($dimensions) {
                    $query->where('width', $dimensions['width'])
                        ->where('height', $dimensions['height']);
                });
            });

        return $query->get();
    }
}
