<?php

namespace Mtc\ContentManager\Contracts;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Facades\Config;
use Mtc\ContentManager\Factories\PageContentFactory;
use Mtc\ContentManager\Traits\HasMedia;

abstract class Content extends Model
{
    use HasMedia;
    use HasFactory;

    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'page_content';

    /**
     * Mass assign attributes
     *
     * @var string[]
     */
    protected $fillable = [
        'template_element_id',
        'global_content_id',
        'element_id',
        'slug',
        'page_id',
        'parent_id',
        'name',
        'content',
        'data',
        'order',
        'hidden',
    ];

    /**
     * Cast attributes to types
     *
     * @var string[]
     */
    protected $casts = [
        'content' => 'array',
        'data' => 'array',
    ];

    /**
     * Hide attributes from json
     *
     * @var string[]
     */
    protected $hidden = [
        'created_at',
        'updated_at',
    ];

    protected static function newFactory()
    {
        return PageContentFactory::new();
    }

    protected static function boot()
    {
        parent::boot();
        self::deleting(fn (self $content) => $content->removeSubContent());
    }

    /**
     * Relationship with content element that defines structure of this element
     *
     * @return BelongsTo
     */
    public function contentElement(): BelongsTo
    {
        return $this->belongsTo(Config::get('pages.element_model'), 'element_id');
    }

    /**
     * Relationship with template element that was used to create this element
     *
     * @return BelongsTo
     */
    public function templateElement(): BelongsTo
    {
        return $this->belongsTo(Config::get('pages.template_element_model'), 'template_element_id');
    }

    /**
     * Relationship with content elements that are nested under this element
     *
     * @return HasMany
     */
    public function subContent(): HasMany
    {
        return $this->hasMany(static::class, 'parent_id')
            ->orderBy('order');
    }

    /**
     * Relationship with content element parent
     *
     * @return BelongsTo
     */
    public function parent(): BelongsTo
    {
        return $this->belongsTo(static::class, 'parent_id');
    }

    /**
     * Relationship with global content element (if global element added instead of normal one)
     *
     * @return BelongsTo
     */
    public function globalContent(): BelongsTo
    {
        return $this->belongsTo(Config::get('pages.global_content_model'), 'global_content_id');
    }

    /**
     * Define thumbnail sizes to auto-generate for this model
     *
     * @return mixed
     */
    public function getDefaultAllowedMediaSizesAttribute()
    {
        return Config::get('pages.thumbnail_sizes', []);
    }

    protected function removeSubContent(): void
    {
        $this->load('subContent');
        $this->subContent->each(fn($content) => $content->delete());
    }
}
