<?php

namespace Mtc\ContentManager;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Mtc\ContentManager\Contracts\Content;
use Mtc\ContentManager\Contracts\ContentElement;
use Mtc\ContentManager\Contracts\ContentElementField;
use Mtc\ContentManager\Contracts\MediaUse;
use Mtc\ContentManager\Contracts\PageModel;
use Mtc\ContentManager\Contracts\TemplateElement;
use Mtc\ContentManager\Contracts\VersionModel;
use Mtc\ContentManager\Facades\Media;
use Mtc\ContentManager\Models\VersionContent;
use Mtc\ContentManager\Traits\EnsuresSlug;

class PageRepository
{
    use EnsuresSlug;

    /**
     * PageRepository constructor.
     * @param PageModel $page
     */
    public function __construct(protected PageModel $pageModel, protected VersionModel $version)
    {
        //
    }

    /**
     * @param string $title
     * @param int|null $templateId
     * @param string|null $category
     * @return PageModel|Model
     */
    public function create(string $title, int|null $templateId, ?string $category = null): PageModel
    {
        if (config('pages.use_transactions')) {
            DB::beginTransaction();
        }
        /** @var PageModel $page */
        $page = $this->pageModel->newQuery()
            ->create([
                'status' => 'draft',
                'title' => $title,
                'slug' => $this->ensureSlug($title, $this->pageModel),
                'template_id' => $templateId,
                'category' => $category,
            ]);

        if ($page->template_id) {
            $this->addContentFromLayout($page);
        }

        if (config('pages.use_transactions')) {
            DB::commit();
        }
        return $page;
    }

    /**
     * Save page data
     *
     * @param PageModel $page
     * @param array $input
     * @param int $author
     */
    public function save(PageModel $page, array $input, int $author): void
    {

        if ($this->setAsPublishedNow($page, $input)) {
            $page->published_at = Carbon::now();
        } else {
            $page->published_at = $input['published_at'] ?? null;
        }
        $page->fill([
            'title' => $input['title'],
            'excerpt' => $input['excerpt'] ?? null,
            'slug' => $input['seo']['slug'] ?? $input['slug'],
            'status' => $input['status'],
            'seo' => $input['seo'] ?? [],
            'meta' => $input['meta'] ?? [],
        ])->save();

        $this->saveCategories($page, $input['categories'] ?? []);
        $this->saveTags($page, $input['tags'] ?? []);
        $this->saveContent($page, $input['content']);

        if ($this->hasPageContentChanged($page, $input['content']) !== false) {
            $this->savePageVersion($page, $input['content'], $author);
        }
    }

    /**
     * Save page data as new version
     *
     * @param VersionModel $versionModel
     * @param array $input
     * @param int $author
     */
    public function saveVersion(VersionModel $versionModel, array $input, int $author): void
    {
        $this->saveContent($versionModel, $input['content']);
    }

    public function canRemove(int $id): bool
    {
        $page = $this->pageModel->newQuery()->find($id);
        return $page && ($page->meta['disallow_delete'] ?? false) !== true;
    }

    /**
     * Mark page as removed
     *
     * @param int|array $id
     * @param bool $force
     * @return mixed
     */
    public function remove(int|array $id, bool $force = false)
    {
        if ($force) {
            return $this->pageModel->newQuery()
                ->whereIn('id', (array)$id)
                ->forceDelete();
        }
        return $this->pageModel->newQuery()
            ->whereIn('id', (array)$id)
            ->get()
            ->filter(fn(PageModel $page) => $page->delete())
            ->count();
    }

    /**
     * Update published state of a page
     *
     * @param int|array $id
     * @param Carbon|null $value
     * @return int
     */
    public function setPublishState(int|array $id, Carbon|null $value)
    {
        return $this->pageModel->newQuery()
            ->whereIn('id', (array)$id)
            ->update([
                'status' => $value ? PageStatus::PUBLISHED->value : PageStatus::DRAFT->value,
                'published_at' => $value
            ]);
    }

    /**
     * Restore a deleted page
     *
     * @param int $pageId
     * @return bool
     */
    public function restorePage(int $pageId): bool
    {
        return $this->pageModel
            ->newQuery()
            ->onlyTrashed()
            ->findOrFail($pageId)
            ->restore();
    }

    /**
     * Restore content from version history
     *
     * @param int $version_id
     */
    public function restoreVersion(int $version_id)
    {
        /** @var VersionModel $version */
        $version = $this->version->newQuery()->findOrFail($version_id);
        $this->setContentFromVersion($version);
        $version->page->versions()->update([
            'is_active' => false,
        ]);
        $version->is_active = true;
        $version->pending_review = false;
        $version->request_changes = false;
        $version->save();
        return $version;
    }

    public function markVersionForReview(int $pageId, int $versionId)
    {
        $version = $this->version->newQuery()
            ->where('page_id', $pageId)
            ->findOrFail($versionId);
        $version->pending_review = true;
        $version->request_changes = false;
        $version->save();
        return $version;
    }

    public function markVersionAsChangesRequested(int $pageId, int $versionId)
    {
        $version = $this->version->newQuery()
            ->where('page_id', $pageId)
            ->findOrFail($versionId);
        $version->pending_review = false;
        $version->request_changes = true;
        $version->save();
        return $version;
    }

    /**
     * Create a copy page
     *
     * @param int $pageId
     * @param string $title
     * @param bool $withContent
     * @return PageModel
     */
    public function copyPage(int $pageId, string $title, bool $withContent)
    {
        /** @var PageModel $original */
        $original = $this->pageModel->newQuery()->findOrFail($pageId);

        if (config('pages.use_transactions')) {
            DB::beginTransaction();
        }
        /** @var PageModel $page */
        $page = $this->pageModel->newQuery()
            ->create([
                'status' => 'draft',
                'title' => $title,
                'category' => $original->category,
                'slug' => Str::slug($title),
                'template_id' => $original->template_id
            ]);

        if ($withContent) {
            $original->content->each(fn(Content $content) => $page->content()->create($content->toArray()));
        } elseif ($page->template_id) {
            $this->addContentFromLayout($page);
        }
        if (config('pages.use_transactions')) {
            DB::commit();
        }
        return $page;
    }

    /**
     * Update page text field with page content
     *
     * @param PageModel $page
     * @return void
     */
    public function updatePageTextField(PageModel $page)
    {
        $page->update([
            'text' => $page->rootLevelContent
                ->keyBy('name')
                ->map(fn(Content $content) => $this->getPageTextContentArray($content))
        ]);
    }

    /**
     * Update page search_content field with page content
     *
     * @param PageModel $page
     * @return void
     */
    public function updatePageSearchContent(PageModel $page)
    {
        return;
        $page->update([
            'search_content' => $page->rootLevelContent
                ->keyBy('name')
                ->map(fn(Content $content) => strip_tags(implode(' ', $this->getPageTextContentArray($content))))
                ->prepend($page->title)
                ->implode(' ')
        ]);
    }

    public function syncContentWithTemplate(int $pageId)
    {
        /** @var PageModel $page */
        $page = $this->pageModel->newQuery()
            ->with([
                'content.templateElement',
                'content.subContent.subContent.subContent.subContent.subContent.subContent',
                'template.elements.element.fields.childElement.fields.childElement'
                . '.fields.childElement.fields.childElement.fields.childElement'
            ])
            ->findOrFail($pageId);

        $this->mergeContentStructure($page, $this->getMissingStructure($page));
        $this->dropTemplateContent($page, $this->getOrphanedElements($page));
        $this->dropTemplateContent($page, $this->getElementsFromDifferentTemplate($page));
        return $page->refresh();
    }

    /**
     *
     * @param PageModel $page
     * @return bool
     */
    public function hasMissingStructure(PageModel $page): bool
    {
        return $this->getMissingStructure($page)->isNotEmpty()
            || $this->getElementsFromDifferentTemplate($page)->isNotEmpty()
            || $this->getOrphanedElements($page)->isNotEmpty();
    }

    public function hasVersioning(): bool
    {
        return true;
    }

    /**
     * Save a new version of the page
     *
     * @param PageModel $page
     * @param array $content
     * @param int $author
     * @param false $is_active
     */
    public function savePageVersion(
        PageModel $page,
        array $content,
        int $author,
        bool $is_active = false
    ): ?VersionModel {
        /** @var VersionModel $version */
        $version = $page->versions()
            ->create([
                'author_id' => $author,
                'is_active' => $is_active,
            ]);
        $this->saveContent($version, $content);
        return $version;
    }

    /**
     * @param PageModel $page
     * @return Collection
     */
    protected function getMissingStructure(PageModel $page): Collection
    {
        if (empty($page->template)) {
            return collect([]);
        }

        return $page->template->elements
            ->filter(fn($element) => $this->isTemplateElementMissingOnPage($element, $page)
                || $this->isElementStructureMissingOnPage($element, $page));
    }


    /**
     * @param PageModel $page
     * @return Collection
     */
    protected function getOrphanedElements(PageModel $page): Collection
    {
        if (empty($page->template_id)) {
            return collect([]);
        }

        return $page->content()
            ->whereDoesntHave('templateElement')
            ->whereNotNull('template_element_id')
            ->get();
    }

    /**
     * Check if template element exists on page
     *
     * @param TemplateElement $element
     * @param PageModel $page
     * @return bool
     */
    protected function isTemplateElementMissingOnPage(TemplateElement $element, PageModel $page): bool
    {
        return $page->content->where('template_element_id', $element->id)->isEmpty();
    }

    protected function isElementStructureMissingOnPage(TemplateElement $templateElement, PageModel $page): bool
    {
        if (!empty($templateElement->global_content_id)) {
            return false;
        }

        return $templateElement->element?->fields
            ->filter(
                fn(ContentElementField $field) => $page->content
                    ->where('template_element_id', $templateElement->id)
                    ->first()
                    ?->subContent->where('slug', $field->slug)
                    ->isEmpty()
            )->isNotEmpty() ?? false;
    }

    /**
     * @param PageModel $page
     * @param Collection $structure
     * @return void
     */
    protected function mergeContentStructure(PageModel $page, Collection $structure)
    {
        $page->content->each(fn(Content $content) => $content->update([
            'order' => $content->templateElement?->order ?? 0,
        ]));
        $structure->each(function ($missingElement) use ($page) {
            $elementData = $missingElement->toArray();
            $elementData['slug'] = $missingElement->global_content_id
                ? $missingElement->globalContent->slug
                : $missingElement->element?->slug;

            /** @var Content $pageContent */
            $pageContent = $page->content()
                ->updateOrCreate([
                    'template_element_id' => $missingElement->id,
                ], $elementData);

            // This is a content element that has fields underneath it
            if ($missingElement->element) {
                $this->nestContentElementsForPage($pageContent, $missingElement->element, true);
            }
        });
    }

    protected function dropTemplateContent(PageModel $page, Collection $structure)
    {
        $structure->each(fn($content) => $content->delete());
    }

    /**
     * Load page content relationships
     *
     * @param PageModel $page
     * @return void
     */
    protected function loadPageContentRelationships(PageModel $page): void
    {
        $page->load([
            'rootLevelContent.globalContent',
            'rootLevelContent.subContent',
            'rootLevelContent.subContent.globalContent',
            'rootLevelContent.subContent.subContent',
            'rootLevelContent.subContent.subContent.globalContent',
            'rootLevelContent.subContent.subContent.subContent.globalContent',
            'rootLevelContent.subContent.subContent.subContent.subContent.globalContent',
        ]);
    }

    /**
     * Structure page content in an array format
     *
     * @param Content $content
     * @return array|mixed
     */
    protected function getPageTextContentArray(Content $content)
    {
        $contentData = $content->global_content_id
            ? $content->globalContent?->content
            : $content?->content;

        if ($contentData === null) {
            $contentData = [];
        }

        if ($content->subContent->isNotEmpty()) {
            $contentData['subContent'] = $content->subContent
                ->keyBy('name')
                ->map(fn(Content $subContent) => $this->getPageTextContentArray($subContent));
        }

        return is_array($contentData) ? $contentData : [$contentData];
    }

    /**
     * Persist page content to database
     *
     * @param PageModel|VersionModel $page
     * @param array $content
     */
    protected function saveContent(PageModel|VersionModel $page, array $content)
    {
        $this->dropOrphanedContent($page, $this->getFlatContentElementIds($content));

        $this->removeNonExistsContent($page, $content);

        collect($content)->each(fn($element, $index) => $page instanceof PageModel
            ? $this->savePageContentElement($page, $element, $index)
            : $this->saveVersionContentElement($page, $element, $index));

        if ($page instanceof PageModel) {
            $this->loadPageContentRelationships($page);
            $this->updatePageTextField($page);
            $this->updatePageSearchContent($page);
        }
    }

    /**
     * Save content element against page or its content element
     *
     * @param PageModel|Content $parent
     * @param array $element
     * @param int $order
     */
    protected function savePageContentElement(PageModel|Content $parent, array $element, int $order): void
    {
        $element['data'] = collect($element)->except([
            'id',
            'name',
            'content',
            'parent_id',
            'page_id',
            'global_content_id',
            'element_id',
            'children',
        ]);

        $element['order'] = $order;

        if (empty($element['page_id'])) {
            $element['page_id'] = $parent instanceof PageModel
                ? $parent->id
                : $parent->page_id;
        }

        /** @var Content $entry */
        $entry = $parent instanceof PageModel
            ? $parent->content()->updateOrCreate(['id' => $element['id'] ?? null], $element)
            : $parent->subContent()->updateOrCreate(['id' => $element['id'] ?? null], $element);

        if (($element['fieldId'] ?? '') === 'image') {
            $this->syncContentElementMedia($element['content'] ?? [], $entry);
        }
        collect($element['children'] ?? [])
            ->each(fn($content, $index) => $this->savePageContentElement($entry, $content, $index));
    }

    /**
     * Save content element against page or its content element
     *
     * @param VersionModel|Content $parent
     * @param array $element
     * @param int $order
     */
    protected function saveVersionContentElement(VersionModel|Content $parent, array $element, int $order): void
    {
        $element['data'] = collect($element)->except([
            'id',
            'name',
            'content',
            'parent_id',
            'owner_id',
            'owner_type',
            'global_content_id',
            'element_id',
            'children',
        ]);
        $element['order'] = $order;

        $element['owner_type'] = 'version';
        if (empty($element['owner_id'])) {
            $element['owner_id'] = $parent instanceof VersionModel
                ? $parent->id
                : $parent->owner_id;
        }

        /** @var Content $entry */
        $entry = $parent instanceof VersionModel
            ? $parent->content()->updateOrCreate(['id' => $element['id'] ?? null], $element)
            : $parent->subContent()->updateOrCreate(['id' => $element['id'] ?? null], $element);

        if (($element['fieldId'] ?? '') === 'image') {
            $this->syncContentElementMedia($element['content'], $entry);
        }
        collect($element['children'] ?? [])
            ->each(fn($content, $index) => $this->saveVersionContentElement($entry, $content, $index));
    }

    /**
     * Set media elements on Content
     *
     * @param array $element
     * @param Content $entry
     * @return void
     */
    protected function syncContentElementMedia(array $media_entries, Content $entry): void
    {
        $ids = collect($media_entries)
            ->map(fn($entry) => is_numeric($entry) ? $entry : $entry['id']);
        $to_remove = $entry->mediaUses
            ->reject(fn($media_use) => $ids->search($media_use['media_id']) !== false)
            ->pluck('id');

        if ($to_remove->isNotEmpty()) {
            $entry->mediaUses()->whereIn('id', $to_remove)->delete();
        }

        $to_add = $ids->filter(fn($id) => $entry->mediaUses->pluck('media_id')->search($id) === false);

        if ($to_add->isNotEmpty()) {
            Media::setUsesForModel($ids->toArray(), $entry);
        }
    }

    /**
     * Check if content on page has changed requiring to save a new version
     *
     * @param PageModel $page
     * @param array $current_content
     * @param int|null $version_id
     * @return bool
     */
    protected function hasPageContentChanged(PageModel $page, array $current_content, int $version_id = null): bool
    {
        $version = $page->versions()
            ->when(
                $version_id,
                fn($query) => $query->where('id', $version_id),
                fn($query) => $query->where('is_active', 1)
            )
            ->first();


        return ($version->content ?? []) !== $current_content;
    }

    /**
     * Create content fields for page based on layout
     *
     * @param PageModel $page
     * @return void
     */
    protected function addContentFromLayout(PageModel $page)
    {
        $page->template->elements()
            ->with([
                'element.fields.childElement.fields.childElement.fields.childElement.fields.childElement'
            ])
            ->get()
            ->each(fn(TemplateElement $element, $index) => $this->saveContentElementForPage($page, $element, $index));
    }

    /**
     * Save a content element against page
     *
     * @param PageModel $page
     * @param TemplateElement $templateElement
     * @param int $oder
     * @return void
     */
    protected function saveContentElementForPage(PageModel $page, TemplateElement $templateElement, int $order)
    {
        /** @var Content $pageContent */
        $pageContent = $page->content()
            ->updateOrCreate(
                ['template_element_id' => $templateElement->id],
                $this->templateElementToContent($templateElement, $order)
            );

        if ($templateElement->element) {
            $this->nestContentElementsForPage($pageContent, $templateElement->element);
        }
    }

    public function addElement(Content $parent, ContentElement $contentElement, array $data = []): Content
    {
        /** @var Content $subContent */
        $subContent = $parent->subContent()
            ->create(
                array_merge([
                    'page_id' => $parent->page_id,
                    'element_id' => $contentElement->id,
                    'slug' => $contentElement->slug,
                    'name' => $contentElement->name,
                    'data' => $contentElement->data,
                ], $data)
            );

        $this->nestContentElementsForPage($subContent, $contentElement);
        return $subContent;
    }

    /**
     * Nest child content elements for a page
     *
     * @param Content $pageContent
     * @param ContentElement $contentElement
     * @param bool $update
     * @return void
     */
    public function nestContentElementsForPage(
        Content $pageContent,
        ContentElement $contentElement,
        bool $update = false
    ): void {
        $contentElement->fields
            ->each(fn(ContentElementField $field, $index) => $this->addContentFieldToPageContent(
                $pageContent,
                $field,
                $index,
                $update
            ));
    }

    /**
     * Add content fields to page content from a nested content field
     *
     * @param Content $pageContent
     * @param ContentElementField $field
     * @param int $order
     * @param bool $update
     * @return void
     */
    protected function addContentFieldToPageContent(
        Content $pageContent,
        ContentElementField $field,
        int $order,
        bool $update = false
    ): void {
        /** @var Content $subContent */
        $subContent = $update
            ? $pageContent->subContent()
                ->updateOrCreate([
                    'page_id' => $pageContent->page_id,
                    'element_id' => $field->child_element_id,
                    'slug' => $field->slug,
                    'parent_id' => $pageContent->id,
                ], [
                    'name' => $field->name,
                    'data' => $this->getContentElementFieldData($field),
                    'order' => $order
                ])
            : $pageContent->subContent()->create([
                'page_id' => $pageContent->page_id,
                'element_id' => $field->child_element_id,
                'parent_id' => $pageContent->id,
                'slug' => $field->slug,
                'name' => $field->name,
                'data' => $this->getContentElementFieldData($field),
                'order' => $order
            ]);

        if ($field->childElement) {
            $this->nestContentElementsForPage($subContent, $field->childElement);

            if (empty($field->data)) {
                $this->setContentDataFromFieldChild($subContent, $field);
            }
        }
    }

    protected function getContentElementFieldData(ContentElementField $field)
    {
        return $field->data;
    }

    /**
     * Set content data from the element that is attached to a field
     *
     * @param Content $subContent
     * @param ContentElementField $field
     * @return void
     */
    protected function setContentDataFromFieldChild(Content $subContent, ContentElementField $field)
    {
        $data = $field->childElement->data ?? [];
        if (!empty($field->childElement->icon)) {
            $data['icon'] = $field->childElement->icon;
        }
        $subContent->update([
            'data' => $data,
        ]);
    }

    /**
     * Convert Template element to Page content object
     *
     * @param TemplateElement $templateElement
     * @param int $order
     * @return array
     */
    protected function templateElementToContent(TemplateElement $templateElement, int $order): array
    {
        $data = $templateElement->toArray();
        $data['template_element_id'] = $templateElement->id;
        if ($templateElement->element_id) {
            $data['data'] = $templateElement->element->data ?? [];
            if (!empty($templateElement->element->icon)) {
                $data['data']['icon'] = $templateElement->element->icon;
            }
        }

        if (empty($data['slug'])) {
            if (!empty($templateElement->data['slug'])) {
                $data['slug'] = $templateElement->data['slug'];
            }
            if (!empty($templateElement->data['fieldId'])) {
                $data['slug'] = $templateElement->data['fieldId'];
            }
        }
        $data['order'] = $order;
        unset($data['id']);
        return $data;
    }

    protected function getFlatContentElementIds(array $content): array
    {
        $ids = collect([]);
        collect($content)->each(function ($entry) use ($ids) {
            if (is_numeric($entry['id'])) {
                $ids->push($entry['id']);
            }
            if (!empty($entry['children'])) {
                $ids->push($this->getFlatContentElementIds($entry['children']));
            }
        });

        return $ids->flatten()->toArray();
    }

    protected function dropOrphanedContent(PageModel|VersionModel $page, array $treeIds)
    {
        $page->allContent()
            ->whereNotIn('id', $treeIds)
            ->delete();
    }

    protected function removeNonExistsContent(
        PageModel|VersionModel $page,
        array &$content = [],
        array $idsToRemove = []
    ): void {
        if (empty($idsToRemove)) {
            $page->load([
                'allContent.contentElement:id',
                'allContent.globalContent:id',
            ]);

            $idsToRemove = $page->allContent
                ->filter(function ($content) {
                    return (
                        (!empty($content->element_id) && empty($content->contentElement))
                        || (!empty($content->global_content_id) && empty($content->globalContent))
                    );
                })
                ->pluck('id')
                ->toArray();

            $page->allContent()
                ->whereIn('id', $idsToRemove)
                ->delete();

            if ($page instanceof PageModel && !empty($page->template)) {
                $page->load([
                    'template.elements.element',
                    'template.elements.globalContent'
                ]);
                foreach ($page->template->elements as $element) {
                    if (
                        (!empty($element->element_id) && empty($element->element))
                        || (!empty($element->global_content_id) && empty($element->globalContent))
                    ) {
                        $element->delete();
                    }
                }
            }
        }

        if (!empty($idsToRemove)) {
            foreach ($content as $key => $element) {
                if (is_numeric($element['id']) && in_array($element['id'], $idsToRemove)) {
                    unset($content[$key]);
                    return;
                }

                if (!empty($element['children'])) {
                    $this->removeNonExistsContent($page, $element['children'], $idsToRemove);
                }
            }
        }
    }

    protected function getElementsFromDifferentTemplate(PageModel $page): Collection
    {
        if (empty($page->template_id)) {
            return collect();
        }

        return $page->content()
            ->whereNotNull('template_element_id')
            ->whereHas('templateElement', fn($query) => $query->where('template_id', '!=', $page->template_id))
            ->get();
    }

    protected function setContentFromVersion(VersionModel $version)
    {
        if (config('pages.use_transactions')) {
            DB::beginTransaction();
        }

        $version->load([
            'content.mediaUses',
            'content.subContent.mediaUses',
            'content.subContent.subContent.mediaUses',
            'content.subContent.subContent.subContent.mediaUses',
            'content.subContent.subContent.subContent.subContent.mediaUses',
            'content.subContent.subContent.subContent.subContent',
        ]);

        $version->page->allContent()->delete();
        $version->content->each(
            fn(VersionContent $versionContent) => $this->setPageContentFromVersionContent(
                $version->page,
                $versionContent,
                null
            )
        );


        if (config('pages.use_transactions')) {
            DB::commit();
        }
    }

    protected function saveCategories(PageModel $page, array $categories): void
    {
        $page->categories()->sync($categories);
    }

    protected function saveTags(PageModel $page, array $tags): void
    {
        $page->tags()->sync($tags);
    }

    protected function setPageContentFromVersionContent(
        PageModel $page,
        VersionContent $versionContent,
        ?int $parentId
    ): void {
        $data = $versionContent->only([
            'template_element_id',
            'global_content_id',
            'element_id',
            'slug',
            'parent_id',
            'name',
            'content',
            'data',
            'order',
        ]);
        $data['parent_id'] = $parentId;
        $pageContent = $page->allContent()->create($data);

        $versionContent->mediaUses
            ->each(function (MediaUse $mediaUse) use ($pageContent) {
                $asArray = $mediaUse->toArray();
                $asArray['owner_type'] = $pageContent->getMorphClass();
                $asArray['owner_id'] = $pageContent->id;
                $pageContent->mediaUses()->create($asArray);
            });

        if (!empty($versionContent->subContent)) {
            $versionContent->subContent->each(
                fn(VersionContent $subContent) => $this->setPageContentFromVersionContent(
                    $page,
                    $subContent,
                    $pageContent->id
                )
            );
        }
    }

    private function setAsPublishedNow(PageModel $page, array $input): bool
    {
        return empty($page->published_at)
            && empty($input['published_at'])
            && $input['status'] === 'published';
    }
}
