<?php

namespace Mtc\ContentManager\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class MediaUploadRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        return Auth::check();
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        return [
            'file' => $this->fileRules(),
            'folder' => [
                'sometimes',
                'exists:media_folders,id'
            ],
        ];
    }

    /**
     * Validation messages for file
     *
     * @return string[]
     */
    public function messages()
    {
        $size_in_mb = $this->fileIsImage() ? 10 : 50;
        return [
            'dimensions' => "Minimal size for image is 100x100",
            'max' => "The file size must be less than $size_in_mb MB.",
        ];
    }

    /**
     * Check if file is an image
     *
     * @return bool
     */
    public function fileIsImage(): bool
    {
        return in_array($this->file('file')?->getMimeType(), config('media.supported_image_mimes'));
    }

    /**
     * Validation rules for file
     *
     * @return string[]
     */
    protected function fileRules()
    {
        if ($this->fileIsImage()) {
            return $this->imageRules();
        }

        return [
            'file',
            'max:50000' // ~50MB
        ];
    }

    /**
     * Validation rules for image
     *
     * @return string[]
     */
    protected function imageRules()
    {
        return [
            'max:10000', // ~10MB
            'mime_types:' . implode(',', config('media.supported_image_mimes')),
            'dimensions:min_width=20,min_height=20'
        ];
    }
}
