<?php

namespace Mtc\ContentManager\Http\Controllers;

use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;
use Mtc\ContentManager\Contracts\GlobalContent;
use Mtc\ContentManager\Http\Requests\CopyGlobalContentElementRequest;
use Mtc\ContentManager\Http\Requests\RemoveGlobalContentRequest;
use Mtc\ContentManager\Http\Requests\StoreGlobalContentRequest;
use Mtc\ContentManager\Http\Requests\UpdateGlobalContentRequest;
use Mtc\ContentManager\Http\Resources\GlobalContentElementView;

class GlobalElementController
{
    use ValidatesRequests;

    /**
     * Show list of global elements
     *
     * @param Request $request
     * @param GlobalContent $model
     * @return LengthAwarePaginator
     */
    public function index(Request $request, GlobalContent $model)
    {
        $perPage = min((int) $request->input('per_page', 15), 200);

        $list = $model->newQuery()
            ->when($request->filled('search_term'), fn($query) => $query->searchTerm($request->input('search_term')))
            ->with('users')
            ->withTemplateCount()
            ->withFreeformPageCount()
            ->withCommentCount()
            ->setSortBy($request->input('sort_by'))
            ->setFilters($request->input('filters') ?? '')
            ->paginate($perPage);

        $resource = Config::get('pages.global_content_list_resource');
        return new $resource($list);
    }

    /**
     * Show a global content element
     *
     * @param GlobalContent $content_block
     * @param int $block_id
     * @return \Illuminate\Http\Response|array
     */
    public function show(GlobalContent $content_block, int $block_id): GlobalContentElementView
    {
        $block = $content_block->newQuery()
            ->with([
                'contentElement.fields',
            ])
            ->find($block_id);

        $resource = Config::get('pages.global_content_view_resource');
        return new $resource($block);
    }

    /**
     * Store a newly created global content element.
     *
     * @param StoreGlobalContentRequest $request
     * @return GlobalContentElementView|Model
     */
    public function store(StoreGlobalContentRequest $request, GlobalContent $globalContent)
    {
        return $globalContent->newQuery()
            ->create($request->input());
    }

    /**
     * Update a global content element
     *
     * @param UpdateGlobalContentRequest $request
     * @param GlobalContent $globalContent
     */
    public function update(UpdateGlobalContentRequest $request, GlobalContent $globalContent, int $elementId)
    {
        if (config('pages.use_transactions')) {
            DB::beginTransaction();
        }
        $element = $globalContent->newQuery()
            ->findOrFail($elementId)
            ?->fill($request->input());

        if ($request->filled('preview_image.path')) {
            $element->preview_image = $request->input('preview_image.path');
        }

        // We don't need other field metadata only id and content
        $element->content = $request->input('content') ?? [];

        $element->save();
        $element->attachContentMedia();

        if (config('pages.use_transactions')) {
            DB::commit();
        }

        $resource = Config::get('pages.global_content_view_resource');
        return new $resource($element);
    }

    /**
     * Remove the specified global content element
     *
     * @param int $id
     * @return void
     */
    public function destroy(RemoveGlobalContentRequest $request, GlobalContent $model, int $id)
    {
        $model->newQuery()->find($id)?->delete();
    }

    /**
     * Create a new copy of an existing global content element
     *
     * @param CopyGlobalContentElementRequest $request
     * @param int $id_to_copy
     * @return GlobalContentElementView
     */
    public function copy(CopyGlobalContentElementRequest $request, GlobalContent $element): GlobalContentElementView
    {
        $copy = $element->newQuery()->create([
            'name' => $request->input('name'),
            'element_id' => $element->element_id,
            'content' => $request->input('with_content') ? $element->content : null,
        ]);

        $resource = Config::get('pages.global_content_view_resource');
        return new $resource($copy);
    }
}
