<?php

namespace Mtc\ContentManager\Traits;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Mtc\ContentManager\Contracts\Content;
use Mtc\ContentManager\Contracts\ContentElementField;
use Mtc\ContentManager\Contracts\PageModel;
use Mtc\ContentManager\Contracts\TemplateElement;
use Mtc\ContentManager\Contracts\VersionContentModel;
use Mtc\ContentManager\Contracts\VersionModel;
use Mtc\ContentManager\Models\VersionContent;
use Mtc\ContentManager\Contracts\ContentElement;

trait VersionContentSync
{
    public function createVersionBackup(PageModel $page, int $author): void
    {
        $version = $page->versions()
            ->create([
                'author_id' => $author,
                'is_active' => true,
            ]);

        $page->content()
            ->with(['mediaUses'])
            ->get()
            ->each(fn(Content $activeContent) => $this->recursivelyCopyContentForVersion($version, $activeContent));
    }

    protected function syncContentWithTemplateToVersion(
        PageModel $page,
        int $author,
        bool $make_active = false
    ): VersionModel {
        /** @var VersionModel $version */
        $version = $page->versions()
            ->create([
                'author_id' => $author,
                'is_active' => $make_active,
            ]);

        if ($page->activeVersion) {
            $this->copyCurrentVersionContent($version, $page->activeVersion);
        } else {
            $this->copyContentFromPage($version, $page);
        }

        $this->elementsOnModelOutOfSync($version)
            ->each(fn(ContentElement $element) => $this->syncSpecificElementOnVersion($version, $element));

        if ($version->page?->template) {
            $this->syncTemplateElementsToVersion($version);
            $this->dropTemplateContent($this->getElementsNoLongerOnTemplate(
                $version,
                $version->page->template->elements
            ));
            $this->dropTemplateContent($this->getElementsFromDifferentTemplate($version));
            $this->dropTemplateContent($this->getOrphanedElements($version));
        }
        return $version;
    }

    protected function copyCurrentVersionContent(VersionModel $new_version, VersionModel $current_version): void
    {
        $current_version->content()
            ->with(['mediaUses'])
            ->get()
            ->each(fn(VersionContentModel $activeContent) => $this->recursivelyCopyContentForVersion(
                $new_version,
                $activeContent
            ));
    }

    protected function recursivelyCopyContentForVersion(
        VersionModel $new_version,
        Content|VersionContentModel $activeContent,
        ?VersionContentModel $parent = null
    ): void {
        $copy = $activeContent->toArray();
        $copy['owner_id'] = $new_version->id;
        $copy['owner_type'] = 'version';
        /** @var VersionContentModel $new_content */
        $new_content = !empty($parent)
            ? $parent->subContent()->create($copy)
            : $new_version->content()->create($copy);

        if ($new_content->slug === 'image') {
            $this->copyContentElementMediaUses($new_content->content ?? [], $new_content);
        }

        $activeContent->subContent
            ->each(fn(Content|VersionContentModel $subContent) => $this->recursivelyCopyContentForVersion(
                $new_version,
                $subContent,
                $new_content
            ));
    }

    protected function removeVersionContentWithMissingTemplateElements(VersionModel $version): void
    {
        $version->content()
            ->whereDoesntHave('templateElement')
            ->whereNotNull('template_element_id')
            ->get()
            ->each(fn($content) => $content->delete());
    }

    public function syncSpecificElement(int $page_id, ContentElement $element): void
    {
        $page = $this->pageModel->newQuery()
            ->with('allContent.subContent')
            ->find($page_id);
        if (!$page) {
            return;
        }

        $page->allContent->where('element_id', $element->id)
            ->each(fn(Content $pageContent, $index) => $element->fields
                ->each(fn(ContentElementField $field) => $this->ensureElementFieldIsOnModelContent(
                    $pageContent,
                    $field,
                    $page,
                    $index
                )))
            ->each(fn(Content $pageContent, $index) => $this->dropRemovedFieldEntriesFromContent(
                $pageContent,
                $element->fields->pluck('slug')
            ));
    }
    protected function syncSpecificElementOnVersion(VersionModel $version, ContentElement $element): void
    {
        $version->allContent->where('element_id', $element->id)
            ->each(fn(VersionContent $pageContent, $index) => $element->fields
                ->each(fn(ContentElementField $field) => $this->ensureElementFieldIsOnModelContent(
                    $pageContent,
                    $field,
                    $version,
                    $index
                )))
            ->each(fn(VersionContent $versionContent, $index) => $this->dropRemovedFieldEntriesFromContent(
                $versionContent,
                $element->fields->pluck('slug')
            ));
    }

    protected function dropRemovedFieldEntriesFromContent(Model $parent_content, Collection $field_slugs): void
    {
        $parent_content->subContent
            ->whereNotIn('slug', $field_slugs)
            ->each(fn(Content $contentEntry) => $contentEntry->delete());
    }

    protected function getElementsNoLongerOnTemplate(VersionModel $version, Collection $templateElements): Collection
    {
        return $version->content()
            ->whereNotNull('template_element_id')
            ->whereNotIn('template_element_id', $templateElements->pluck('id'))
            ->get();
    }

    protected function syncTemplateElementsToVersion(VersionModel $version): void
    {
        $version->page?->template?->elements
            ->each(function (TemplateElement $template_element, $order) use ($version) {
                $on_model_content = $this->ensureTemplateElementIsInModelContent($version, $template_element, $order);
                if ($template_element->element) {
                    $template_element->element->fields
                        ->each(fn(ContentElementField $field, $order) => $this->ensureElementFieldIsOnModelContent(
                            $on_model_content,
                            $field,
                            $version,
                            $order
                        ));
                }
            });
    }

    protected function copyContentFromPage(VersionModel $new_version, PageModel $page): void
    {
        $page->content()
            ->with(['mediaUses'])
            ->get()
            ->each(fn(Content $activeContent) => $this->recursivelyCopyContentForVersion(
                $new_version,
                $activeContent
            ));
    }

    protected function elementsOnModelOutOfSync(VersionModel $version): Collection
    {
        return $version->allContent()
            ->with('contentElement')
            ->whereHas('contentElement')
            ->get()
            ->filter(fn(VersionContent $content) => $this->pageElementFieldsDoNotMatchTemplate(
                $content,
                $content->contentElement
            ));
    }
}
