<?php

namespace Mtc\ContentManager\Http\Controllers;

use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Mtc\ContentManager\Contracts\ContentElement;
use Mtc\ContentManager\Contracts\ElementRepositoryContract;
use Mtc\ContentManager\Http\Requests\CopyContentElementRequest;
use Mtc\ContentManager\Http\Requests\StoreContentElementRequest;
use Mtc\ContentManager\Http\Requests\UpdateContentElementRequest;
use Mtc\ContentManager\Http\Resources\ContentElementList;
use Mtc\ContentManager\Http\Resources\ContentElementView;

class ElementController
{
    use ValidatesRequests;

    /**
     * Display a listing of content elements in system.
     *
     * @param Request $request
     * @param ElementRepositoryContract $repository
     * @return ContentElementList
     */
    public function index(Request $request, ElementRepositoryContract $repository): ContentElementList
    {
        $perPage = min((int) $request->input('per_page', 25), 200);

        $resource = Config::get('pages.element_list_resource');
        return new $resource(
            $repository->getList(
                $request->input('search-term', ''),
                $request->input('sort_by', ''),
                $perPage,
                false,
                true,
                false
            )
        );
    }

    /**
     * Show the content element in detail
     *
     * @param int $elementId
     * @param ElementRepositoryContract $repository
     * @return ContentElementView
     */
    public function show(int $elementId, ElementRepositoryContract $repository): ContentElementView
    {
        $resource = Config::get('pages.element_view_resource');
        return new $resource($repository->find($elementId));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param StoreContentElementRequest $request
     * @param ElementRepositoryContract $repository
     * @return ContentElementView
     */
    public function store(
        StoreContentElementRequest $request,
        ElementRepositoryContract $repository
    ): ContentElementView {
        $element = $repository->create(
            title: $request->input('title'),
        );
        $resource = Config::get('pages.element_view_resource');
        return new $resource($element);
    }

    /**
     * Update content element
     *
     * @param UpdateContentElementRequest $request
     * @param ElementRepositoryContract $repository
     * @return ContentElementView
     */
    public function update(
        UpdateContentElementRequest $request,
        ContentElement $element,
        ElementRepositoryContract $repository
    ): ContentElementView {

        if (config('pages.use_transactions')) {
            DB::beginTransaction();
        }
        $element = $repository->update($element, $request->input());

        if ($request->hasFile('icon_file')) {
            Storage::disk('content-icons')
                ->put($request->file('icon_file')->getClientOriginalName(), $request->file('icon_file'));
        }

        if (config('pages.use_transactions')) {
            DB::commit();
        }

        $resource = Config::get('pages.element_view_resource');
        return new $resource($element);
    }

    /**
     * Create a new copy of an existing content element
     *
     * @param CopyContentElementRequest $request
     * @param int $id_to_copy
     * @param ElementRepositoryContract $repository
     * @return ContentElementView
     */
    public function copy(
        CopyContentElementRequest $request,
        int $id_to_copy,
        ElementRepositoryContract $repository
    ): ContentElementView {
        $element = $repository->makeCopy($id_to_copy, $request->input('title'));
        $resource = Config::get('pages.element_view_resource');
        return new $resource($element);
    }

    /**
     * Remove the content element.
     *
     * @param int $elementId
     * @param ElementRepositoryContract $repository
     * @return array
     */
    public function destroy(int $elementId, ElementRepositoryContract $repository): array
    {
        return [
            'status' => $repository->destroy($elementId) ? 'success' : 'failed',
        ];
    }

    /**
     * Retrieve element usage - check element before saving to prevent breaking changes.
     * Usage limits 10 records per type
     *
     * @param int $elementId
     * @param ElementRepositoryContract $repository
     * @return int[]
     */
    public function checkUsage(int $elementId, ElementRepositoryContract $repository): array
    {
        return $repository->getUsage($elementId);
    }

    /**
     * Show full usage of a content element for the content type
     *
     * @param int $elementId
     * @param string $type
     * @param ElementRepositoryContract $repository
     * @return Collection
     */
    public function fullUsageList(int $elementId, string $type, ElementRepositoryContract $repository): Collection
    {
        return $repository->fullUsageList($elementId, $type);
    }

    /**
     * Show list of icons available for content elements
     *
     * @return array
     */
    public function icons(): array
    {
        return Storage::disk('content-icons')->allFiles();
    }
}
