<?php

namespace Mtc\ContentManager\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\Rule;
use Mtc\ContentManager\PageStatus;
use Mtc\ContentManager\Rules\IsSlugRule;
use Mtc\ContentManager\Rules\MaxNumericValue;
use Mtc\ContentManager\Rules\MinNumericValue;
use Mtc\ContentManager\Rules\PageContentHasAllBlueprintFields;

class UpdatePageRequest extends FormRequest
{
    protected $extraAttributes = [];
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        return Auth::check();
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        return array_merge([
            'title' => 'required',
            'seo.slug' => [
                'nullable',
                new IsSlugRule(),
                'unique:pages,slug,' . ($this->page->id ?? 0)
            ],
            'seo.description' => [
                'nullable',
                'max:500',
            ],
            'seo.title' => [
                'nullable',
                'max:250',
            ],
            'published_at' => 'sometimes',
            'status' => [
                'required',
                Rule::in(array_map(static fn (PageStatus $status) => $status->value, PageStatus::cases())),
            ],
            'seo' => 'array',
            'seo.jsonSchema' => [
                'nullable',
                'json',
            ],
            'content' => [
                'array',
            ],
            'content.*.id' => [
                'present',
            ],
            'content.*.element_id' => [
                'nullable',
                'numeric',
            ],
            'content.*.global_content_id' => [
                'nullable',
                'numeric',
            ],
            'content.*.content' => [
                'nullable',
            ],
            'content.*' => [
                'array',
            ],
        ], $this->getContentFieldRules());
    }

    public function attributes()
    {
        return array_merge([
            'seo.jsonSchema' => 'JSON schema',
            'seo.slug' => 'page slug',
            'seo.description' => 'page seo description',
            'seo.title' => 'page seo title',
        ], $this->extraAttributes);
    }

    protected function getContentFieldRules(): array
    {
        return $this->recursiveContentMetaOptionsCheck($this->input('content', []), 'content');
    }

    protected function recursiveContentMetaOptionsCheck(array $content, $prefix = '', $namePrefix = ''): array
    {
        $rules = [];
        foreach ($content as $index => $entry) {
            $nameValue = collect([
                $namePrefix,
                $entry['title'] ?? $entry['name'] ?? ''
            ])->filter()->implode(' > ');
            if (!empty($entry['meta']['options'])) {
                $field = "$prefix.$index.content";
                foreach ($entry['meta']['options'] as $rule => $value) {
                    $rules[$field] = collect($entry['meta']['options'])
                        ->map(fn ($value, $rule) => $this->convertToValidationRules($rule, $value))
                        ->filter()
                        ->toArray();
                }
                $this->extraAttributes[$field] = $nameValue;
            }

            if (!empty($entry['children'])) {
                $rules = array_merge(
                    $rules,
                    $this->recursiveContentMetaOptionsCheck($entry['children'], "$prefix.$index.children", $nameValue)
                );
            }
        }

        return $rules;
    }

    private function convertToValidationRules(string $rule, mixed $value)
    {
        return match ($rule) {
            'required' => $value ? 'required' : '',
            'max' => new MaxNumericValue($value),
            'min' => new MinNumericValue($value),
            'minLength' => "min:$value",
            'maxCount', 'maxLength' => "max:$value",
            default => '',
        };
    }
}
