<?php

namespace Mtc\ContentManager\Http\Resources;

use Carbon\Carbon;
use Illuminate\Http\Resources\Json\JsonResource;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Auth;
use Mtc\ContentManager\Contracts\MediaUse;
use Mtc\ContentManager\Models\ContentCategory;
use Mtc\ContentManager\Models\ContentTag;
use Mtc\ContentManager\Models\Template;
use Mtc\ContentManager\PageRepository;
use Mtc\ContentManager\Traits\ContentMetaValues;
use Mtc\ContentManager\Traits\MapContentTrait;
use Mtc\ContentManager\Models\ContentElement;
use Mtc\ContentManager\Models\GlobalContent;

class PageViewResource extends JsonResource
{
    use MapContentTrait;
    use ContentMetaValues;

    public static $wrap = '';

    /**
     * Transform the resource into an array.
     *
     * @param \Illuminate\Http\Request $request
     * @return array|\Illuminate\Contracts\Support\Arrayable|\JsonSerializable
     */
    public function toArray($request)
    {
        $this->resource->load([
            'template.elements.element.fields',
            'mediaUses.media',
            'content.mediaUses.media',
            'content.subContent.mediaUses.media',
            'content.subContent.subContent.mediaUses.media',
            'content.subContent.subContent.subContent.mediaUses.media',
            'content.subContent.subContent.subContent.subContent.mediaUses.media',
            'content.globalContent.contentElement.fields',
            'content.contentElement.fields',
            'content.subContent.contentElement.fields',
            'content.subContent.subContent.contentElement.fields',
            'content.subContent.subContent.subContent.contentElement.fields',
            'content.subContent.subContent.subContent.subContent.contentElement.fields',
            'content.subContent.subContent.subContent.subContent.subContent.contentElement.fields',
        ]);

        $templates = Template::select(['id','name'])->active()->get();
        $categories = ContentCategory::select(['id','name'])->get();
        $tags = ContentTag::select(['id','name'])->get();
        $this->resource->template_name = $templates->where('id', $this->resource->template_id)->first()?->name;

        $resource = $this->resource->toArray();

        $resource['tags'] = $this->resource->tags()->pluck('tag_id');
        $resource['categories'] = $this->resource->categories()->pluck('category_id');
        $resource['structure_in_sync'] = $this->pageRepository()->hasMissingStructure($this->resource) !== true;
        $resource['media'] = $this->resource->mediaUses->pluck('media_id');
        $resource['media_uses'] = $this->resource->mediaUses
            ->map(fn(MediaUse $mediaUse) => new MediaEntryResource($mediaUse));
        $resource['content'] = $this->resource->content
            ->filter(fn($content) => !empty($content->slug)
                || !empty($content->template_element_id)
                || !empty($content->element_id)
                || !empty($content->global_content_id))
            ->map(fn ($content) => $this->mapContent($content))
            ->values();
        $resource['user_can_edit'] = $this->userCanEditResource();
        $resource['editableContentStructure'] = $this->template_id === null
            || ($this->resource->template?->meta && $this->resource->template?->meta['flexible'] ?? false);
        $resource['meta'] = $this->getMetaValues();
        $resource['versions'] = $this->getVersionData();

        unset($resource['text']);

        if ($resource['editableContentStructure'] === false) {
            return [
                'page' => $resource,
                'templates' => $templates,
                'categories' => $categories,
                'tags' => $tags,
                'global_content' => [],
                'elements' => [],
            ];
        }

        $blockResource = config('pages.content_element_block_resource');
        $elements = ContentElement::query()
            ->with('fields.childElement.fields.childElement.fields.childElement.fields')
            ->where('is_enabled', 1)
            ->where('drag_and_drop', 1)
            ->paginate(10);

        return [
            'page' => $resource,
            'templates' => $templates,
            'categories' => $categories,
            'tags' => $tags,
            'global_content' => new $blockResource(GlobalContent::query()->paginate(10)),
            'elements' => new $blockResource($elements),
        ];
    }

    protected function mapVersion($pageVersion)
    {
        return [
            'title' => $pageVersion->id . ($pageVersion->is_active ? ' (Active)' : ''),
            'description' => __('pages.version_description', [
                'author' => $pageVersion->author->name,
                'date' => Carbon::parse($pageVersion->created_at)->format('d/m/Y H:i'),
            ]),
            'id' => $pageVersion->id,
            'uuid' => $pageVersion->uuid,
            'status' => $pageVersion->status,
            'comments' => $pageVersion->comment_count,
            'is_active' => $pageVersion->is_active,
            'created_at' => Carbon::parse($pageVersion->created_at)->format('d/m/Y H:i'),
            'url' => $pageVersion->url(),
            'author' => $pageVersion->author_id
                ? [ 'alt' => $pageVersion->author->name, 'src' => $pageVersion->author->profile_photo_url ]
                : [],
        ];
    }

    private function pageRepository(): PageRepository
    {
        return App::make(config('pages.page_repository'));
    }

    private function userCanEditResource(): bool
    {
        return true;
    }

    private function getVersionData()
    {
        if ($this->pageRepository()->hasVersioning() !== true) {
            return false;
        }

        return $this->resource->versions()
            ->withCommentCount()
            ->with([
                'author',
                'page',
            ])
            ->latest()
            ->get()
            ->map(fn ($version) => $this->mapVersion($version));
    }
}
