<?php

namespace Mtc\ContentManager;

use Exception;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;
use Mtc\ContentManager\Facades\Media;

class ImageSize
{
    /**
     * ImageSize constructor
     *
     * @param int $width
     * @param int $height
     * @param bool $allow_compression
     */
    public function __construct(
        protected int $width = 0,
        protected int $height = 0,
        protected bool $allow_compression = true
    ) {
    }

    /**
     * Create instance from array
     *
     * @param array $config
     * @return static
     */
    public static function fromArray(array $config): self
    {
        return new self($config['width'] ?? 0, $config['height'] ?? 0, $config['allow_compression'] ?? true);
    }

    /**
     * Find destination for size on path
     *
     * @param string $fileName
     * @return string
     * @throws Exception
     */
    public function fileDestination(string $fileName, ?string $path = null, ?bool $legacyFormat = false): string
    {
        return Storage::disk(Config::get('filesystems.default_media'))
            ->path($this->pathOnDisk($fileName, $path, $legacyFormat));
    }

    /**
     * Find the file url for size
     *
     * @param string $fileName
     * @return string
     */
    public function fileUrl(string $fileName, ?string $path = '', ?bool $legacyFormat = false): string
    {
        return Storage::disk(Config::get('filesystems.default_media'))
            ->url($this->pathOnDisk($fileName, $path, $legacyFormat));
    }

    /**
     * Find the path on disk for the file
     *
     * @param string $fileName
     * @return string
     */
    public function pathOnDisk(string $fileName = '', string $path = '', ?bool $legacyFormat = false): string
    {
        return $legacyFormat
            ?  rtrim(Media::storagePrefix(), '/') . "/{$this->width}x$this->height/$fileName"
            : rtrim(str_replace("/orig/", "/{$this->width}x$this->height/", $path), '/') . "/$fileName";
    }

    /**
     * Get width of the ImageSize
     *
     * @return int
     */
    public function getWidth(): int
    {
        return $this->width;
    }

    /**
     * Get height of the ImageSize
     *
     * @return int
     */
    public function getHeight(): int
    {
        return $this->height;
    }

    /**
     * return whether size allows compression on image when resizing
     *
     * @return bool
     */
    public function getAllowCompression(): bool
    {
        return $this->allow_compression;
    }
}
